--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]

local ffi           = require[[ffi]]
local sys           = require[[lua_system\\system]]
local wnd           = require[[lua_system\\window]]
local key           = require[[lua_system\\key_code]]

local user32        = ffi.load("user32")

local kbd           = {}


-- layout = getlayout([handle])
-- получает активную раскладку клавиатуры
--
-- layout - раскладка клавиатуры
--
-- handle - хэндл окна.
--          По умолчанию window.
do
    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE    HKL;
        typedef unsigned int       DWORD;
        HKL GetKeyboardLayout(
            DWORD idThread
        );
    ]]
    kbd.getlayout = function(handle)
        handle = handle or window
        local _, tid, err, verbose = wnd.windowpid(handle)
        if not tid then return nil, err, verbose end
        local result = user32.GetKeyboardLayout(tid)
        return result
    end
end


-- setlayout(new_layout)
-- Меняет активную раскладку.
--
-- new_layout - новая раскладка. Может быть
--              задана хэндлом либо числом.
--
-- Примечание:
-- Ошибки нормально не выбрасывает,
-- можно вытащить результат
-- через DefWindowProc, смысла
-- реализовывать на данный момент не вижу.
do
    local WM_INPUTLANGCHANGEREQUEST = 0x50

    kbd.setlayout = function(hkl, handle)
        handle = handle or window
        -- Отправить сообщение WM_INPUTLANGCHANGEREQUEST с новой раскладкой
        wnd.postmessage(WM_INPUTLANGCHANGEREQUEST,
                        0,
                        ffi.cast("int", hkl),
                        handle)
    end
end


-- layout, err, verbose = getlayoutlist()
-- Получает список всех раскладок
-- клавиатуры в системе.
--
-- layout  - таблица с хэндлами раскладок
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
do
    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE    HKL;
        int GetKeyboardLayoutList(int nBuff, HKL *lpList);
    ]]

    local buffer_size = 4
    local buffer = ffi.new("HKL[?]", buffer_size)

    -- Функция для получения списка доступных раскладок
    kbd.getlayoutlist = function()
        local layouts = {}

        local copied = 0
        repeat
            local success = true
            -- Получаем раскладки.
            copied = user32.GetKeyboardLayoutList(buffer_size, buffer)
            -- Вызов GetKeyboardLayoutList с нулевым
            -- размером буфера по докам должен
            -- вернуть количество раскладок в системе
            -- без копирования в буфер.
            -- Но функции плевать на маны, ругается
            -- на недопустимые права доступа к памяти,
            -- если задать 0.
            -- Делаем вызовы и увеличиваем буфер пока
            -- количество скопированных раскладок
            -- не станет меньше размера буфера.
            if     copied == 0 then
                return nil, sys.geterr()
            elseif copied == buffer_size then
                -- Выделяем память под массив дескрипторов раскладок
                buffer_size = 2^math.ceil(math.log(buffer_size+1, 2))
                buffer = ffi.new("HKL[?]", buffer_size)
                success = false
            end
        until success

        -- Записываем дескрипторы раскладок в lua aтаблицу
        for i = 0, copied - 1 do
            local layout = buffer[i]
            layouts[i + 1] = layout
        end

        return layouts, 0, "ERROR_SUCCESS"
    end
end


-- state, toggle = keystate(button)
-- Получает состояние нажата ли клавиша
--
-- state  - true - нажата, false - нет.
--
-- toggle - горит ли лампочка кнопки
--          актуально только для
--          num_lock, caps_lock, scroll_lock,
--          для других клавиш при каждом нажатии
--          меняет свое состояние
--          если было true, то на false,
--          если было false, то на true.
--
-- button - клавиша для проверки.
--          Может быть задана строкой или числом.
--          Строка указывает на имя клавиши,
--          например, "ctrl", "a", "1".
--          Если строка задана числом,
--          то вернет состояние виртуальной
--          клавиши с данным номером.
--          Внимание!
--          keystate(1) - задано числом, вернет
--          состояние клавиши с кодом 1
--          (левая кнопка мыши)
--          keystate("1") - вернет состояние
--          клавиши 1 на клавиатуре.
--          Подробнее ознакомиться со списком
--          клавиш можно в файле key_code.lua.
do
    ffi.cdef[[
        typedef          short     SHORT;
        SHORT GetKeyState(
            int nVirtKey
        );
    ]]
    kbd.keystate = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        local state = user32.GetKeyState(button)
        return bit.band(state, 0x8000) ~= 0, bit.band(state, 1) ~= 0
    end
end


-- state, toggle = keystateasync(button)
-- Получает состояние нажата ли клавиша
--
-- state  - true - нажата, false - нет.
--
-- toggle - было ли нажатие клавиши с момента
--          прошлого вызова keystateasync.
--
-- button - клавиша для проверки.
--          Может быть задана строкой или числом.
--          Строка указывает на имя клавиши,
--          например, "ctrl", "a", "1".
--          Если строка задана числом,
--          то вернет состояние виртуальной
--          клавиши с данным номером.
--          Внимание!
--          keystate(1) - задано числом, вернет
--          состояние клавиши с кодом 1
--          (левая кнопка мыши)
--          keystate("1") - вернет состояние
--          клавиши 1 на клавиатуре.
--          Подробнее ознакомиться со списком
--          клавиш можно в файле key_code.lua.
do
    ffi.cdef[[
        typedef          short     SHORT;
        SHORT GetAsyncKeyState(
            int vKey
        );
    ]]
    kbd.keystateasync = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        local state = user32.GetAsyncKeyState(button)
        return bit.band(state, 0x8000) ~= 0, bit.band(state, 1) ~= 0
    end
end


-- key(button, [mod_1, [mod_2, ...]], [handle])
-- Нажимает клавишу button, с возможностью
-- зажатых модификаторов.
--
-- button - клавиша которая должна быть нажата.
--
-- mod_*  - модификатор который должен быть
--          зажат во время нажатия button.
--          Предполагается возможное зажатие
--          "shift", "ctrl", "alt", но
--          не ограничиваясь ими.
--          Количество не ограничено.
--          Подробнее ознакомиться со списком
--          клавиш можно в файле key_code.lua.
--
-- handle - хэндл окна в котрое посылать нажатие.
--          Только для send и post.
--
--
-- down(button, system, [handle])
-- Зажать клавишу button.
--
-- button - клавиша которая должна быть зажата.
--
-- system - только для send и post.
--          флаг того, что клавиша должна
--          быть зажата системно с использованием
--          WM_SYSKEYDOWN.
--          Возможные значения: nil/true/false.
--          По умолчанию nil.
--
-- handle - хэндл окна в котрое посылать нажатие.
--          Только для send и post.
--
--
-- up(button, system, [handle])
-- Отжать клавишу button.
--
-- button - клавиша которая должна быть отжата.
--
-- system - только для send и post.
--          флаг того, что клавиша должна
--          быть отжата системно с использованием
--          WM_SYSKEYUP.
--          Возможные значения: nil/true/false.
--          По умолчанию nil.
--
-- handle - хэндл окна в котрое посылать нажатие.
--          Только для send и post.
--
--
-- text(text, [handle])
-- Написать текст text.
--
-- text   - текст для ввода.
--          Если среда выполнения не является
--          редактором uopilot версии 2.42 и ниже,
--          то текст должен быть в utf8.
--
-- handle - хэндл окна в котрое посылать нажатие.
--          Только для send и post.
--
--
-- Примечания.
-- send, post методы могут отправлять сочетания
-- клавиш не зажимая alt, ctrl, shift для всей
-- системы. При этом может наблюдаться отсутствие
-- эффекта при комбинировании с реальной клавитурой.
-- Т.е. если программно зажать alt через send или post,
-- а затем нажать F4 на реально клавиатуре - блокнот
-- не закроется. При этом полностю программная эмуляция
-- post("F4", "alt") прекрасно работает.
-- keystate("alt") так же будет говорить что alt не нажат.
--
-- при использвании down и up в send и post
-- они будут пытаться автоматически определить
-- необходимость эмулировать системное нажатие
-- из-за использованя alt и F10.
-- Тем не менее настоятельно рекомендуются использовать
-- их симметрично. Т.е. если вы нажимали что-то
-- через down, то эта кнопка должна быть отжата
-- через up, а не другим методом или физической
-- клавиатурой. Аналогично не стоит пытаться
-- отжать с помощью up клавиши которые вы не нажимали
-- с помощью down.
-- В противном случае может возникнуть путаница
-- с флагами системного нажатия.
-- Если все-таки очень хочется, то:
-- 1) выше описанные проблемы касаются исключительно
-- сочетаний клавиш с ctrl, alt и F10. Перепутанный флаг
-- ctrl может повлиять только на alt. Если alt не используете,
-- то последствий быть не должно.
-- 2) при работе с alt и F10 прямо задавайте флаг
-- системного нажатия либо обычного.
-- Логика работы винды при обработке системных клавиш:
--[[if keydown then
        if key == alt
            if ctrl_pressed then
                msg = 0x100
            else
                msg = 0x104
            end
        elseif key == f10 then
            msg = 0x104
        else
            if alt_pressed then
                msg = 0x104
            else
                msg = 0x100
            end
        end
    end
    if keyup then
        if key == alt then
            -- none_nonsys_was_unpressed
            -- именно было ли ли отжатие
            -- какой-либо не системной клавиши
            -- после нажатия альта.
            -- Это НЕ текущее состояние каких-либо клавиш.
            if ctrl_was_pressed_before_alt then
                msg = 0x101
            else
                if none_nonsys_was_unpressed or ctrl_pressed then
                    msg = 0x105
                else
                    msg = 0x101
                end
            end
        else
            if alt_pressed or key == F10 then
                msg = 0x105
            else
                msg = 0x101
            end
        end
    end
]]


-- message
-- message
-- message
do
    -- Внутренняя функци для создания очереди
    -- кликов с конвертацией текста в нужную
    -- раскладку, переключением раскладок хоткеями,
    -- зажатием shift, ctrl, alt.
    local text_to_message = function(text)end
    do
        ffi.cdef[[
            typedef          void*     PVOID;
            typedef          PVOID     HANDLE;
            typedef          HANDLE    HKL;
            typedef          short     SHORT;
            typedef          char      CHAR;
            typedef unsigned int       UINT;

            SHORT VkKeyScanExA(
                CHAR ch,
                HKL  dwhkl
            );
            UINT MapVirtualKeyA(
                UINT uCode,
                UINT uMapType
            );
        ]]

        local WM_KEYDOWN                = 0x0100
        local WM_KEYUP                  = 0x0101
        local WM_SYSKEYDOWN             = 0x0104
        local WM_SYSKEYUP               = 0x0105
        local MAPVK_VK_TO_VSC           = 0x00  -- scan code
        local VK_SHIFT                  = 0x10  -- shift
        local VK_CONTROL                = 0x11  -- ctrl
        local VK_MENU                   = 0x12  -- alt
        local VK_F10                    = 0x79  -- F10
        local SCAN_SHIFT                = 0x2A  -- shift
        local SCAN_CONTROL              = 0x1D  -- ctrl
        local SCAN_MENU                 = 0x38  -- alt
        local lparam_shift_down         = bit.lshift(SCAN_SHIFT, 16) + 1
        local lparam_ctrl_down          = bit.lshift(SCAN_CONTROL, 16) + 1
        local lparam_alt_down           = bit.lshift(SCAN_MENU, 16) + 1
        local lparam_shift_up           = bit.bor(bit.lshift(SCAN_SHIFT, 16), 0xC0000001)
        local lparam_ctrl_up            = bit.bor(bit.lshift(SCAN_CONTROL, 16), 0xC0000001)
        local lparam_alt_up             = bit.bor(bit.lshift(SCAN_MENU, 16), 0xC0000001)
        local WM_INPUTLANGCHANGEREQUEST = 0x50  -- смена раскладки
        local layout                    = kbd.getlayoutlist()

        text_to_message = function(text)
            text = sys.ispilot and text or sys.utf8_to_ascii(text)

            local start_state_shift = kbd.keystate(VK_SHIFT)
            local start_state_ctrl  = kbd.keystate(VK_CONTROL)
            local start_state_alt   = kbd.keystate(VK_MENU)
            local state_shift       = start_state_shift
            local state_ctrl        = start_state_ctrl
            local state_alt         = start_state_alt

            -- Не в каждой раскладке возможна
            -- отправка всех символов.
            -- Может потребоваться переключение
            -- раскладки во время ввода.
            local vk             = nil
            local scan_code      = nil
            local mod_shift      = nil
            local mod_ctrl       = nil
            local mod_alt        = nil
            local key_queue      = {}
            local layout_current = sys.indexof(layout, kbd.getlayout())
            local layout_prev    = layout_current

            for i = 1, #text do
                local ch = text:sub(i, i):byte()

                local scan_mask = user32.VkKeyScanExA(ch, layout[layout_current])

                -- Маска не получена, перебираем раскладки.
                while scan_mask < 1 do
                    if  layout_current ~= #layout then
                        layout_current  =  layout_current + 1
                    else
                        layout_current = 1
                    end

                    if  layout_current == layout_prev then
                        return nil, -1, "layouts have no symbol '"..text:sub(i, i).."', pos: "..i
                    end
                    scan_mask = user32.VkKeyScanExA(ch, layout[layout_current])
                end
                if layout_prev ~= layout_current then
                    key_queue[#key_queue+1] = {WM_INPUTLANGCHANGEREQUEST,
                                               0,
                                               ffi.cast("int", layout[layout_current])}
                    layout_prev = layout_current
                end

                -- vk символа
                vk = bit.band(scan_mask, 0xFF)
                scan_code = ffi.C.MapVirtualKeyA(vk, MAPVK_VK_TO_VSC)

                -- Парсим битовую маску модификаторов.
                mod_shift = bit.band(scan_mask, 2^ 8) ~= 0
                mod_ctrl  = bit.band(scan_mask, 2^ 9) ~= 0
                mod_alt   = bit.band(scan_mask, 2^10) ~= 0

                -- Добавляем зажатие/отжатие модификаторов
                -- CTRL
                if  mod_ctrl and not state_ctrl then
                    state_ctrl = true
                    key_queue[#key_queue+1] = {not state_alt and WM_KEYDOWN or WM_SYSKEYDOWN,
                                               VK_CONTROL,
                                               lparam_ctrl_down}
                elseif not mod_ctrl and state_ctrl then
                    state_ctrl = false
                    key_queue[#key_queue+1] = {not state_alt and WM_KEYUP or WM_SYSKEYUP,
                                               VK_CONTROL,
                                               lparam_ctrl_up}
                end

                -- ALT
                if  mod_alt and not state_alt then
                    state_alt = true
                    key_queue[#key_queue+1] = {not state_alt and WM_KEYDOWN or WM_SYSKEYDOWN,
                                               VK_MENU,
                                               lparam_alt_down}
                elseif not mod_alt and state_alt then
                    state_alt = false
                    key_queue[#key_queue+1] = {not state_ctrl and WM_KEYUP or WM_SYSKEYUP,
                                               VK_MENU,
                                               lparam_alt_up}
                end

                -- SHIFT
                if  mod_shift and not state_shift then
                    state_shift = true
                    key_queue[#key_queue+1] = {not state_alt and WM_KEYDOWN or WM_SYSKEYDOWN,
                                               VK_SHIFT,
                                               lparam_shift_down}
                elseif not mod_shift and state_shift then
                    state_shift = false
                    key_queue[#key_queue+1] = {not state_alt and WM_KEYUP or WM_SYSKEYUP,
                                               VK_SHIFT,
                                               lparam_shift_up}
                end


                -- Собственно клавиша.
                key_queue[#key_queue+1] = { not state_alt and WM_KEYDOWN or WM_SYSKEYDOWN,
                                            vk,
                                            bit.lshift(scan_code, 16) + 1}

                key_queue[#key_queue+1] = {}
                if not state_alt then
                    if vk ~= VK_F10 then
                        key_queue[#key_queue][1] = WM_KEYUP
                    else
                        key_queue[#key_queue][1] = WM_SYSKEYUP
                    end
                else
                    key_queue[#key_queue][1] = WM_SYSKEYUP
                end
                key_queue[#key_queue][2] = vk
                key_queue[#key_queue][3] = bit.bor(bit.lshift(scan_code, 16), 0xC0000001)
            end

            -- Отжимаем все модификаторы в конце работы

            -- CTRL
            if  state_ctrl then
                key_queue[#key_queue+1] = {not state_alt and WM_KEYUP or WM_SYSKEYUP,
                                           VK_CONTROL,
                                           lparam_ctrl_up}
            end
            -- ALT
            if  state_alt then
                key_queue[#key_queue+1] = {not state_ctrl and WM_KEYUP or WM_SYSKEYUP,
                                           VK_MENU,
                                           lparam_alt_up}
            end
            -- SHIFT
            if  state_shift then
                key_queue[#key_queue+1] = {not state_alt and WM_KEYUP or WM_SYSKEYUP,
                                           VK_SHIFT,
                                           lparam_shift_up}
            end

            return key_queue, 0, "ERROR_SUCCESS"
        end
    end


    -- message
    -- message
    -- message
    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE    HKL;
        typedef          short     SHORT;
        typedef          char      CHAR;
        typedef unsigned int       UINT;

        UINT MapVirtualKeyA(
            UINT uCode,
            UINT uMapType
        );
        SHORT VkKeyScanExA(
            CHAR ch,
            HKL  dwhkl
        );
    ]]

    local WM_KEYDOWN         = 0x0100
    local WM_KEYUP           = 0x0101
    local WM_SYSKEYDOWN      = 0x0104
    local WM_SYSKEYUP        = 0x0105
    local VK_CTRL            = 0x0011
    local VK_MENU            = 0x0012  -- alt
    local VK_F10             = 0x0079
    local mask_alt_flag      = 2^29
    local MAPVK_VK_TO_VSC    = 0x00000000

    -- Список методов отправки сообщений.
    local message_method   = {}
    message_method.send    = wnd.sendmessage
    message_method.post    = wnd.postmessage

    -- кратко про конченную логику winapi
--[[if keydown then
        if key == alt
            if ctrl_pressed then
                msg = 0x100
            else
                msg = 0x104
            end
        elseif key == f10 then
            msg = 0x104
        else
            if alt_pressed then
                msg = 0x104
            else
                msg = 0x100
            end
        end
    end
    if keyup then
        if key == alt then
            -- none_nonsys_was_unpressed
            -- именно было ли ли отжатие
            -- какой-либо не системной клавиши
            -- после нажатия альта.
            -- Это НЕ текущее состояние каких-либо клавиш.
            if ctrl_was_pressed_before_alt then
                msg = 0x101
            else
                if none_nonsys_was_unpressed or ctrl_pressed then
                    msg = 0x105
                else
                    msg = 0x101
                end
            end
        else
            if alt_pressed or key == F10 then
                msg = 0x105
            else
                msg = 0x101
            end
        end
    end
]]
    for method, msg_func in pairs(message_method) do
        kbd[method]= {}

        kbd[method].text = function(text, handle)
            local queue = text_to_message(text, msg_func)
            local result = nil
            for i = 1, #queue do
                result = msg_func(queue[i][1], queue[i][2], queue[i][3], handle)
            end

            return result
        end


        local state_alt  = false
        local state_ctrl = false
        local nonsys_was_unpressed = false


        kbd[method].key = function(button, ...)
            if type(button) == "string" then
                button = key[button]
            end

            local handle = nil
            -- Нажимаем модификаторы
            local mod = {...}
            for i = 1, #mod do
                if type(mod[i]) == "string" then
                    kbd[method].down(mod[i])
                else
                    handle = mod[i]
                end
            end

            local result_down = kbd[method].down(button, nil, handle)
            local result_up   = kbd[method].up(button, nil, handle)

            -- Отжимаем модификаторы
            for i = 1, #mod do
                if type(mod[i]) == "string" then
                    kbd[method].up(mod[i])
                end
            end

            return result_down, result_up
        end


        kbd[method].down = function(button, param1, param2)
            if type(button) == "string" then
                button = key[button]
            end

            local system_flag = type(param1) ~= "cdata" and param1 or param2
            local handle      = type(param1) == "cdata" and param1 or param2 or window

            local msg = nil
            if system_flag == nil then
                if button == VK_MENU then
                    if state_ctrl then
                        msg = WM_KEYDOWN
                    else
                        msg = WM_SYSKEYDOWN
                        -- state_alt в данном случае
                        -- выражает флаг является ли
                        -- процесс обработки сочетания
                        -- системным.
                        -- Если первым был нажат ctrl
                        -- обработка не системная.
                        -- Если первым был нажат alt
                        -- обработка системная.
                        state_alt = true
                        nonsys_was_unpressed = false
                    end
                elseif button == VK_F10 then
                    msg = WM_SYSKEYDOWN
                else
                    if state_alt then
                        msg = WM_SYSKEYDOWN
                    else
                        msg = WM_KEYDOWN
                    end
                    if button == VK_CTRL then
                        state_ctrl = true
                    end
                end
            elseif system_flag == false then
                msg = WM_KEYDOWN
            elseif system_flag == true then
                msg = WM_SYSKEYDOWN
            end

            local scan = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)

            local lparam = nil
            if not state_alt then
                lparam = bit.lshift(scan, 16) + 1
            else
                lparam = bit.lshift(scan, 16) + 1
                lparam = bit.bor(lparam, mask_alt_flag)
            end

            local result = msg_func(msg, button,
                                    lparam,
                                    handle)
            return result
        end


        kbd[method].up = function(button, param1, param2)
            if type(button) == "string" then
                button = key[button]
            end

            local system_flag = type(param1) ~= "cdata" and param1 or param2
            local handle      = type(param1) == "cdata" and param1 or param2 or window

            local msg = nil
            if system_flag == nil then
                if button == VK_MENU then
                    if nonsys_was_unpressed then
                        msg = WM_KEYUP
                    else
                        -- state_alt в данном случае
                        -- выражает флаг является ли
                        -- процесс обработки сочетания
                        -- системным.
                        -- Если первым был нажат ctrl
                        -- обработка не системная.
                        -- Если первым был нажат alt
                        -- обработка системная.
                        if state_alt then
                            msg = WM_SYSKEYUP
                        end
                    end
                    state_alt = false
                else
                    if state_alt or key == VK_F10 then
                        msg = WM_SYSKEYUP
                    else
                        msg = WM_KEYUP
                    end
                    nonsys_was_unpressed = true
                end
            end

            local scan = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)

            local lparam = nil
            if not state_alt then
                lparam = bit.lshift(scan, 16) + 1
                lparam = bit.bor(lparam, 0xC0000001)
            else
                lparam = bit.lshift(scan, 16) + 1
                lparam = bit.bor(lparam, mask_alt_flag, 0xC0000001)
            end
print(msg, button,
                                    lparam,
                                    handle)
            local result = msg_func(msg, button,
                                    lparam,
                                    handle)
            return result
        end
    end
end

-- Внутренняя функци для создания очереди
-- кликов с конвертацией текста в нужную
-- раскладку, переключением раскладок хоткеями,
-- зажатием shift, ctrl, alt.
local text_to_key_queue = function(text)end
do
    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE    HKL;
        typedef          short     SHORT;
        typedef          char      CHAR;
        typedef unsigned int       UINT;

        SHORT VkKeyScanExA(
            CHAR ch,
            HKL  dwhkl
        );
        UINT MapVirtualKeyA(
            UINT uCode,
            UINT uMapType
        );
    ]]

    local MAPVK_VK_TO_VSC    = 0x00000000
    local VK_SHIFT           = 0x10  -- shift
    local VK_CONTROL         = 0x11  -- ctrl
    local VK_MENU            = 0x12  -- alt
    local VK_TILDE           = 0xC0  -- тильда
    local SCAN_SHIFT         = 0x2A  -- shift
    local SCAN_CONTROL       = 0x1D  -- ctrl
    local SCAN_MENU          = 0x38  -- alt
    local SCAN_TILDE         = 0x29  -- тильда
    local DOWN               = 0x00  -- нажать
    local UP                 = 0x02  -- отжать флаг KEYEVENTF_KEYUP
    local layout             = kbd.getlayoutlist()
    local register = sys.registryread(
        [[HKEY_CURRENT_USER\Keyboard Layout\Toggle\Language Hotkey]])
    local switch_lang_hotkey = register and tonumber(register) or 1

    local switch_lang = {}
    switch_lang[1]    = {
            {VK_MENU,    SCAN_MENU,    DOWN},
            {VK_SHIFT,   SCAN_SHIFT,   DOWN},
            {VK_MENU,    SCAN_MENU,    UP},
            {VK_SHIFT,   SCAN_SHIFT,   UP},
        }
    switch_lang[2]    = {
            {VK_CONTROL, SCAN_CONTROL, DOWN},
            {VK_SHIFT,   SCAN_SHIFT,   DOWN},
            {VK_CONTROL, SCAN_CONTROL, UP},
            {VK_SHIFT,   SCAN_SHIFT,   UP},
        }
    switch_lang[4]    = {
            {VK_TILDE,   SCAN_TILDE,   DOWN},
            {VK_TILDE,   SCAN_TILDE,   UP},
        }

    text_to_key_queue = function(text)
        text = sys.ispilot and text or sys.utf8_to_ascii(text)

        local start_state_shift = kbd.keystate(VK_SHIFT)
        local start_state_ctrl  = kbd.keystate(VK_CONTROL)
        local start_state_alt   = kbd.keystate(VK_MENU)
        local state_shift       = start_state_shift
        local state_ctrl        = start_state_ctrl
        local state_alt         = start_state_alt

        -- Не в каждой раскладке возможна
        -- отправка всех символов.
        -- Может потребоваться переключение
        -- раскладки во время ввода.
        local vk             = nil
        local scan_code      = nil
        local mod_shift      = nil
        local mod_ctrl       = nil
        local mod_alt        = nil
        local key_queue      = {}
        local layout_current = sys.indexof(layout, kbd.getlayout())
        local layout_prev    = layout_current

        for i = 1, #text do
            local ch = text:sub(i, i):byte()

            local scan_mask = user32.VkKeyScanExA(ch, layout[layout_current])

            -- Маска не получена, перебираем раскладки.
            while scan_mask < 1 do
                if  layout_current ~= #layout then
                    layout_current  =  layout_current + 1
                else
                    layout_current = 1
                end

                for j = 1, #switch_lang[switch_lang_hotkey] do
                    key_queue[#key_queue+1] = switch_lang[switch_lang_hotkey][j]
                end

                if  layout_current == layout_prev then
                    return nil, -1, "layouts have no symbol '"..text:sub(i, i).."', pos: "..i
                end
                scan_mask = user32.VkKeyScanExA(ch, layout[layout_current])
            end
            layout_prev = layout_current

            -- vk символа
            vk = bit.band(scan_mask, 0xFF)
            scan_code = ffi.C.MapVirtualKeyA(vk, MAPVK_VK_TO_VSC)

            -- Парсим битовую маску модификаторов.
            mod_shift = bit.band(scan_mask, 2^ 8) ~= 0
            mod_ctrl  = bit.band(scan_mask, 2^ 9) ~= 0
            mod_alt   = bit.band(scan_mask, 2^10) ~= 0

            -- Добавляем зажатие/отжатие модификаторов
            -- CTRL
            if  mod_ctrl and not state_ctrl then
                state_ctrl = true
                key_queue[#key_queue+1] = {VK_CONTROL, SCAN_CONTROL, DOWN}
            elseif not mod_ctrl and state_ctrl then
                state_ctrl = false
                key_queue[#key_queue+1] = {VK_CONTROL, SCAN_CONTROL, UP}
            end

            -- ALT
            if  mod_alt and not state_alt then
                state_alt = true
                key_queue[#key_queue+1] = {VK_MENU, SCAN_MENU, DOWN}
            elseif not mod_alt and state_alt then
                state_alt = false
                key_queue[#key_queue+1] = {VK_MENU, SCAN_MENU, UP}
            end

            -- SHIFT
            if  mod_shift and not state_shift then
                state_shift = true
                key_queue[#key_queue+1] = {VK_SHIFT, SCAN_SHIFT, DOWN}
            elseif not mod_shift and state_shift then
                state_shift = false
                key_queue[#key_queue+1] = {VK_SHIFT, SCAN_SHIFT, UP}
            end

            -- Прожимаем целевую кнопку.
            key_queue[#key_queue+1] = {vk, scan_code, DOWN}
            key_queue[#key_queue+1] = {vk, scan_code, UP}
        end

        -- Отжимаем все модификаторы в конце работы

        -- CTRL
        if  state_ctrl then
            key_queue[#key_queue+1] = {VK_CONTROL, SCAN_CONTROL, UP}
        end
        -- ALT
        if  state_alt then
            key_queue[#key_queue+1] = {VK_MENU,    SCAN_MENU,    UP}
        end
        -- SHIFT
        if  state_shift then
            key_queue[#key_queue+1] = {VK_SHIFT,   SCAN_SHIFT,   UP}
        end

        return key_queue, 0, "ERROR_SUCCESS"
    end
end


-- event
-- event
-- event
do
    if ffi.arch == "x64" then
        ffi.cdef[[
                typedef unsigned __int64 ULONG_PTR;
            ]]
    else
        ffi.cdef[[
                typedef unsigned long ULONG_PTR;
            ]]
    end
    ffi.cdef[[
        typedef unsigned int       UINT;
        typedef unsigned char      BYTE;
        typedef unsigned int       DWORD;

        UINT MapVirtualKeyA(
            UINT uCode,
            UINT uMapType
        );
        void keybd_event(
            BYTE      bVk,
            BYTE      bScan,
            DWORD     dwFlags,
            ULONG_PTR dwExtraInfo
        );
    ]]
    local MAPVK_VK_TO_VSC = 0x00000000
    local KEYEVENTF_KEYUP = 0x0002

    kbd.event = {}

    kbd.event.text = function(text)
        local queue = text_to_key_queue(text)
        for i = 1, #queue do
            user32.keybd_event(queue[i][1], queue[i][2], queue[i][3], 0)
        end
    end


    kbd.event.key = function(button, ...)
        if type(button) == "string" then
            button = key[button]
        end

        -- Нажимаем модификаторы
        local mod = {...}
        for i = 1, #mod do
            if not kbd.keystate(mod[i]) then
                kbd.event.down(mod[i])
            end
        end

        local scan = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        local result_down = user32.keybd_event(button, scan, 0, 0)
        local result_up   = user32.keybd_event(button, scan, KEYEVENTF_KEYUP, 0)

        -- Отжимаем модификаторы
        for i = 1, #mod do
            kbd.event.up(mod[i])
        end

        return result_down, result_up
    end


    kbd.event.down = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        local scan = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        local result = user32.keybd_event(button, scan, 0, 0)

        return result
    end


    kbd.event.up = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        local scan = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        local result = user32.keybd_event(button, scan, KEYEVENTF_KEYUP, 0)

        return result
    end
end


-- input
-- input
-- input
do
    if ffi.arch == "x64" then
        ffi.cdef[[
            typedef unsigned __int64 ULONG_PTR;
        ]]
    else
        ffi.cdef[[
            typedef unsigned long ULONG_PTR;
        ]]
    end
    ffi.cdef[[
        typedef          long      LONG;
        typedef unsigned int       DWORD;
        typedef unsigned short     WORD;
        typedef unsigned int       UINT;

        UINT MapVirtualKeyA(
            UINT uCode,
            UINT uMapType
        );
    ]]
    sys.define[[
        typedef struct tagMOUSEINPUT {
            LONG      dx;
            LONG      dy;
            DWORD     mouseData;
            DWORD     dwFlags;
            DWORD     time;
            ULONG_PTR dwExtraInfo;
        } MOUSEINPUT, *PMOUSEINPUT, *LPMOUSEINPUT;
    ]]
    sys.define[[
        typedef struct tagKEYBDINPUT {
            WORD      wVk;
            WORD      wScan;
            DWORD     dwFlags;
            DWORD     time;
            ULONG_PTR dwExtraInfo;
        } KEYBDINPUT, *PKEYBDINPUT, *LPKEYBDINPUT;
    ]]
    sys.define[[
        typedef struct tagHARDWAREINPUT {
            DWORD uMsg;
            WORD  wParamL;
            WORD  wParamH;
        } HARDWAREINPUT, *PHARDWAREINPUT, *LPHARDWAREINPUT;
    ]]
    sys.define[[
    typedef struct tagINPUT {
        DWORD type;
        union {
            MOUSEINPUT    mi;
            KEYBDINPUT    ki;
            HARDWAREINPUT hi;
        };
    } INPUT, *PINPUT, *LPINPUT;
    ]]
    ffi.cdef[[
        UINT SendInput(
            UINT    cInputs,
            LPINPUT pInputs,
            int     cbSize
        );
    ]]
    local MAPVK_VK_TO_VSC         = 0x00000000

    --local INPUT_MOUSE           = 0
    local INPUT_KEYBOARD          = 1
    --local INPUT_HARDWARE        = 2


    --local KEYEVENTF_EXTENDEDKEY = 0x0001
    local KEYEVENTF_KEYUP         = 0x0002
    --local KEYEVENTF_SCANCODE    = 0x0008
    --local KEYEVENTF_UNICODE     = 0x0004

    local buffer_size             = 0
    local buffer                  = nil
    local input_size              = ffi.sizeof("INPUT")
    kbd.input                  = {}

    -- Функция создания и инициализации
    -- статичных элементов буфера.
    local check_buffer_size = function(data_size)
        if  buffer_size < data_size then
            buffer_size = 2^math.ceil(math.log(data_size, 2))
            buffer      = ffi.new("INPUT[?]", buffer_size)
            for i = 0, buffer_size - 1 do
                buffer[i].type             = INPUT_KEYBOARD
                buffer[i].ki.time        = 0
                buffer[i].ki.dwExtraInfo = 0
            end
        end
    end
    -- Инициализируем буфер.
    check_buffer_size(2048)


    kbd.input.text = function(text)
        local queue = text_to_key_queue(text)

        -- Увеличиваем буфер, если необходимо
        check_buffer_size(#queue)

        -- Создаем адаптиврованную очередь под input
        for i = 0, #queue-1 do
            buffer[i].ki.wVk     = queue[i+1][1]
            buffer[0].ki.wScan   = queue[i+1][2]
            buffer[i].ki.dwFlags = queue[i+1][3]
        end

        local result = user32.SendInput(#queue, buffer, input_size)

        return result
    end


    kbd.input.key = function(button, ...)
        if type(button) == "string" then
            button = key[button]
        end

        -- Нажимаем модификаторы
        local mod = {...}
        for i = 1, #mod do
            if not kbd.keystate(mod[i]) then
                kbd.input.down(mod[i])
            end
        end

        buffer[0].type           = INPUT_KEYBOARD
        buffer[0].ki.wVk         = button
        buffer[0].ki.wScan       = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        buffer[0].ki.dwFlags     = 0
        buffer[0].ki.time        = 0
        buffer[0].ki.dwExtraInfo = 0

        buffer[1].type           = INPUT_KEYBOARD
        buffer[1].ki.wVk         = button
        buffer[1].ki.wScan       = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        buffer[1].ki.dwFlags     = KEYEVENTF_KEYUP
        buffer[1].ki.time        = 0
        buffer[1].ki.dwExtraInfo = 0

        local result = user32.SendInput(2, buffer, input_size)

        -- Отжимаем модификаторы
        for i = 1, #mod do
            kbd.input.up(mod[i])
        end

        return result
    end


    kbd.input.down = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        buffer[0].type           = INPUT_KEYBOARD
        buffer[0].ki.wVk         = button
        buffer[0].ki.wScan       = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        buffer[0].ki.dwFlags     = 0
        buffer[0].ki.time        = 0
        buffer[0].ki.dwExtraInfo = 0

        local result = user32.SendInput(1, buffer, input_size)

        return result
    end


    kbd.input.up = function(button)
        if type(button) == "string" then
            button = key[button]
        end
        buffer[0].type           = INPUT_KEYBOARD
        buffer[0].ki.wVk         = button
        buffer[0].ki.wScan       = ffi.C.MapVirtualKeyA(button, MAPVK_VK_TO_VSC)
        buffer[0].ki.dwFlags     = KEYEVENTF_KEYUP
        buffer[0].ki.time        = 0
        buffer[0].ki.dwExtraInfo = 0

        local result = user32.SendInput(1, buffer, input_size)

        return result
    end
end

return kbd




















