--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]

local ffi           = require[[ffi]]
local sys           = require[[lua_system\\system]]
local kbd           = require[[lua_system\\keyboard]]
local wnd           = require[[lua_system\\window]]

local user32        = ffi.load("user32")

local mouse         = {}


-- x, y, other_pos, err, verbose = parse_crds(params)
-- Парсит координаты в порядке приоритета:
-- crds_table[1].x         crds_table[1].y
-- crds_table[1][1]        crds_table[1][2]
-- crds_table[1][1].x      crds_table[1][1].y
-- crds_table[1][1][1]     crds_table[1][1][2]
--
-- x         - x найденная координата
--
-- y         - y найденная координата
--
-- other_pos - позиция с которой начинаются
--             остальные параметры.
--             Фактически 2 или 3.
--
-- err       - код ошибки. 0 - успех.
--
-- verbose   - текстовая расшифровка ошибки
local parse_crds = function(param)
    if type(param[1]) == "table" then
        if param[1].x and param[1].y then
            return param[1].x, param[1].y, 2, 0, "ERROR_SUCCESS"
        else
            if type(param[1][1]) == "number" then
                return param[1][1], param[1][2], 2, 0, "ERROR_SUCCESS"
            else
                if param[1][1].x and param[1][1].x then
                    return param[1][1].x, param[1][1].y, 2, 0, "ERROR_SUCCESS"
                else
                    return param[1][1][1], param[1][1][2], 2, 0, "ERROR_SUCCESS"
                end
            end
        end
    else
        return param[1], param[2], 3, 0, "ERROR_SUCCESS"
    end
    return nil, nil, nil, -1, "can't find crds"
end


-- x, y = mousepos(abs_flag)
-- Получение текущих координат курсора.
--
-- x - x координата курсора
--
-- y - y координата курсора
--
-- abs - флаг того, что координаты
--       нужны абсолютные
do
    ffi.cdef[[
        typedef          long      LONG;
    ]]
    sys.define[[
        typedef struct tagPOINT {
            LONG x;
            LONG y;
        } POINT, *PPOINT, *NPPOINT, *LPPOINT;
    ]]
    ffi.cdef[[
        typedef          int       BOOL;
        BOOL GetPhysicalCursorPos(
            LPPOINT lpPoint
        );
    ]]
    local point = ffi.new("POINT")

    mouse.mousepos = function(abs_flag)
        local result = user32.GetPhysicalCursorPos(point)
        if result == 0 then return nil, nil, sys.geterr() end
        if not abs_flag then
            local pos, err, verbose = wnd.windowpos()
            if not pos then return nil, err, verbose end
            return point.x-pos.x, point.y-pos.y, 0, "ERROR_SUCCESS"
        else
            return point.x, point.y, 0, "ERROR_SUCCESS"
        end
    end
end


-- Клик мышью.
-- Функции и параметры методов соответственно
-- содержатся подмассивах методов:
-- send
-- post
-- event
-- input
-- Функции кликов:
-- move              (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- left              (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- left_down         (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- left_up           (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- left_double       (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- right             (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- right_down        (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- right_up          (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- right_double      (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- middle            (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- middle_down       (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- middle_up         (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- middle_double     (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m4                (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m4_down           (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m4_up             (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m4_double         (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m5                (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m5_down           (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m5_up             (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- m5_double         (<x, y>|<crds_table>, ["abs"], [modifer], [handle])
-- wheel_up          (<x, y>|<crds_table>, ["abs"], [count], [modifer], [handle])
-- wheel_down        (<x, y>|<crds_table>, ["abs"], [count], [modifer], [handle])
-- wheel_left        (<x, y>|<crds_table>, ["abs"], [count], [modifer], [handle])
-- wheel_right       (<x, y>|<crds_table>, ["abs"], [count], [modifer], [handle])
--
-- Координаты могут быть заданы напрямую двумя числами x и y,
-- либо таблицей (для прямой совместимости с результатми findimage и findcolor)
-- x                 - x координата клика
--
-- y                 - y координата клика
--
-- crds_table        - таблица в которой содержатся x и y координаты.
--                     Таблица анализируется согласно типам данных,
--                     координатами признаются в порядке убывания
--                     приоритета:
--                     crds_table.x         crds_table.y
--                     crds_table[1]        crds_table[2]
--                     crds_table[1].x      crds_table[1].y
--                     crds_table[1][1]     crds_table[1][2]
--
-- "abs"             - флаг того, что координаты абсолютные.
--                     Т.е. x=0, y=0 - верхний левый угол экрана,
--                     а не окна.
--
-- count             - количество прокруток колесом
--                     (только для wheel_* функций)
--
-- modifer           - массив с модификаторами
--                     указывающий какие клавиши
--                     зажаты при клике.
--                     Допустимые значения:
--                     "shift"
--                     "ctrl"
--                     "alt"
--                     "move"        (только для event и input)
--                     "left_down"
--                     "left_up"     (только для event и input)
--                     "right_down"
--                     "right_up"    (только для event и input)
--                     "middle_down"
--                     "middle_up"   (только для event и input)
--                     только один из группы одновременно:
--                     "m4_down"
--                     "m4_up"       (только для event и input)
--                     "m5_down"
--                     "m5_up"       (только для event и input)
--                     "wheel_down"  (только для event и input)
--                     "wheel_up"    (только для event и input)
--                     "wheel_right" (только для event и input)
--                     "wheel_left"  (только для event и input)
--
-- handle            - хэндл окна. По умолчанию window.
--                     Если указан "abs",
--                     то handle игнорируется.
--
--
-- Каждый метод имеет свои внутренние
-- таймауты в комбинированных кликах (down+up)
-- (left, right, middle, mouse4, mouse5).
--
-- timeout_move      - явная пауза между move в точку
--                     клика и событием down.
--                     Имеет смысл только при
--                     По умолчанию 0.
--
-- timeout_inner     - явная пауза между down и up.
--                     По умолчанию 10.
--
-- timeout_outer     - пауза между up и down.
--                     Фактически это пауза между
--                     двумя вызовами, например, парой left
--                     или между left и right.
--                     Данная пауза является не блокирующей,
--                     т.е. если вы совершили некторый клик,
--                     и выполняете иные действия в коде
--                     не связанные с кликами, то таймер все
--                     равно продолжает исчислятся,
--                     вплоть до нуля, что привидет к тому,
--                     что следующий клик будет совершен без
--                     каких либо дополнительных задержек.
--                     По умолчанию 10.
--
-- clock_up          - время прошлого события up в
--                     комбинированных кликах. Используется
--                     для рассчета таймера на следующий клик.
--                     При комбинированных кликах обновляется
--                     автоматически.
--                     Можно использовать для неблокирующей
--                     дополнительной задержки пред
--                     следующим кликом. Например:
--                     clock_up = os.clock() + 2
--                     укажет, что следующий клик должен
--                     произойти не ранее чем через
--                     2 секунды + timeout_outer
--                     с текущего момента.
--
-- Опции методов:
--
-- move_split        - Перемещение и клик могут быть
--                     объедены либо разделены в один
--                     вызов WinAPI функции
--                     (работает только для event и input).
--                     При выключенном разделении
--                     timeout_move будет проигнорирован.
--                     true - разделить, false - объеденить.
--                     По умолчанию false.
--
-- move_before_click - Включить перемещение мыши
--                     перед кликом.
--                     event и input методы фактически
--                     кликают только в текущую позицию
--                     курсора, move необходим им
--                     для корректной работы.
--                     По умолчанию для event и input true,
--                     для остальных методов - false.
--
-- move_after_click  - Возвращать мышь после клика
--                     на исходную позицию.
--                     По умолчанию false.
--
-- move_func         - функция для перемещения мыши, которая
--                     будет использована при
--                     move_before_click и move_after_click.
--                     Наиболее актуально для send и post
--                     методов, т.к. встроенный move очень
--                     специфичный и может не работать с
--                     некоторыми приложениями.
--                     По умолчанию функция из того же метода,
--                     что и клик (send.move для send.left и т.д.).
--
-- mod_func          - массив с функциями которые будут
--                     использованы для зажатия модификаторов
--                     shift, ctrl, alt.
--                     Внимание! Это именно массив, а не функция.
--                     Массив должен содержать функции down и up.
--                     По умолчанию:
--                     send  -> kbd.send
--                     post  -> kbd.send (!)
--                     event -> kbd.event
--                     input -> kbd.input
--
-- mod_inner         - send и post имеют встроенный функционал
--                     для передачи клика с флагом shift, ctrl,
--                     но не alt. Если эта опция включена -
--                     send и post будут использовать именно
--                     внутренние флаги для shift и ctrl, для
--                     alt будет использована mod_func.
--                     Если выключено, то для всех модификаторов
--                     клавиатуры будет использован mod_func.
--                     Возможные значения: true/false.
--                     Значение по умолчанию false.
--
-- Пример вызова:
-- left_down (100, 200, "abs")
-- left_up   (100, 200, {"ctrl", "shift"})
-- left      (100, 200, "abs", {"ctrl", "shift"})
-- wheel_down(100, 200,     2, {"ctrl", "shift"})
--
--
-- Примечание.
-- Все нижеследующие ограничения обусловлены
-- реализацией WinAPI, а не текущим модулем.
--
--
-- Для SEND и POST:
-- Все действия производятся с окном напрямую
-- не затрагивая реальный курсор и клавиатуру
-- (за исключением модификатора "alt").
--
-- Модификатор "alt" в явном виде не поддерживается.
-- Для его зажатия используется эмуляция клавиатуры.
-- Т.е. клавиша будет зажата для всей системы.
-- Время изменения состояния минимизировано,
-- но это может создавать помехи при работе в других
-- приложениях, а ручное нажатие/отжатие alt с
-- некоторой будет вероятностью сбивать клик.
--
-- Данная функция не поддерживает abs.
-- Клик в абсолютные координаты эмулируется
-- вызовом windowfrompoint, windowpos,
-- рассчетом смещения и явной передачей хэндла.
--
-- Количество прокруток за один вызов
-- не должно привышать 273.
--
--
-- Для EVENT, INPUT:
--
-- Все действия влияют на реальный курсор мыши.
--
-- Данная функция не поддерживает handle.
-- Клик в относительыне координаты является
-- вызовом windowpos, рассчетом смещения.
--
-- Модификаторы "shift", "ctrl", "alt"
-- в явном виде не поддерживаются.
-- Для их зажатия используется эмуляция клавиатуры.
-- Т.е. клавиши будет зажаты для всей системы.
do
    -- Константы wparam для mouse4 и mouse5.
    local XBUTTON1         = 0x00010000
    local XBUTTON2         = 0x00020000

    -- Список модификоторов которые
    -- могут быть отправлены,
    -- как зажатые клавиши/кнопки.
    local modifer_list     = {
        left_down          = 0x0001, -- MK_LBUTTON  The left mouse button is down.
        right_down         = 0x0002, -- MK_RBUTTON  The right mouse button is down.
        shift              = 0x0004, -- MK_SHIFT    The SHIFT key is down.
        ctrl               = 0x0008, -- MK_CONTROL  The CTRL key is down..
        middle_down        = 0x0010, -- MK_MBUTTON  The middle mouse button is down
        m4_down            = 0x0020, -- MK_XBUTTON1 The first X button is down.
        m5_down            = 0x0040, -- MK_XBUTTON2 The second X button is down.
    }

    -- Типы базовых кликов предоставляемых WinAPI.
    -- Константы сообщений
    local click_type       = {
        move               = 0x0200, -- WM_MOUSEMOVE
        left_down          = 0x0201, -- WM_LBUTTONDOWN
        left_up            = 0x0202, -- WM_LBUTTONUP
        left_double        = 0x0203, -- WM_LBUTTONDBLCLK
        right_down         = 0x0204, -- WM_RBUTTONDOWN
        right_up           = 0x0205, -- WM_RBUTTONUP
        right_double       = 0x0206, -- WM_RBUTTONDBLCLK
        middle_down        = 0x0207, -- WM_MBUTTONDOWN
        middle_up          = 0x0208, -- WM_MBUTTONUP
        middle_double      = 0x0209, -- WM_MBUTTONDBLCLK
        wheel_up           = 0x020A, -- WM_MOUSEWHEEL
        wheel_down         = 0x020A, -- WM_MOUSEWHEEL
        m4_down            = 0x020B, -- WM_XBUTTONDOWN
        m4_up              = 0x020C, -- WM_XBUTTONUP
        m4_double          = 0x020D, -- WM_XBUTTONDBLCLK
        m5_down            = 0x020B, -- WM_XBUTTONDOWN
        m5_up              = 0x020C, -- WM_XBUTTONUP
        m5_double          = 0x020D, -- WM_XBUTTONDBLCLK
        wheel_left         = 0x020E, -- WM_MOUSEHWHEEL
        wheel_right        = 0x020E, -- WM_MOUSEHWHEEL
    }

    -- Список методов отправки сообщений.
    local message_method   = {}
    message_method.send    = wnd.sendmessage
    message_method.post    = wnd.postmessage

    -- Парсер параметров
    local param_parser     = function(direction, mod_inner, ...)
        local handle           = nil
        local modifer          = 0
        local keyboard_modifer = {}
        local param            = {...}

        -- Проверяем координаты заданны массивом
        -- или переменными.
        -- start_i - стартовая позиция парсинга
        -- массива с остальными параметрами.
        -- Первые позиции обрабатываются
        -- спцифически (координаты)
        local x, y, start_i, err, verbose = parse_crds(param)
        if err ~= 0 then return nil, nil, nil, nil, err, verbose end

        for i = start_i, #param do
            if     type(param[i]) == "string" then
                -- Координаты абсолютные.
                if param[i] == "abs"             then
                    -- abs_flag
                    -- Координаты абсолютные.
                    -- Рассчитываем offset,
                    -- явно указываем handle.
                    -- В данном методе клик может
                    -- быть только в конкретное окно,
                    -- abs - эмуляция.
                    -- Задаем окно.
                    local handle = wnd.windowfrompoint(x, y)
                    local pos    = wnd.windowpos(handle)
                    x = x - pos.x
                    y = y - pos.y
                elseif param[i] == "left_down"
                    or param[i] == "right_down"
                    or param[i] == "middle_down"
                    or param[i] == "m4_down"
                    or param[i] == "m5_down"     then
                    modifer     =   bit.bor(modifer, modifer_list[param[i]])
                elseif param[i] == "shift"
                    or param[i] == "ctrl"        then
                        if mod_inner then
                            modifer = bit.bor(modifer, modifer_list[param[i]])
                        else
                            keyboard_modifer[#keyboard_modifer+1] = param[i]
                        end
                elseif param[i] == "alt"         then
                    keyboard_modifer[#keyboard_modifer+1] = param[i]
                end
            elseif type(param[i]) == "number" then
                -- Вся прокрутка идет с шагом 120.
                -- Виндовые ограничения.
                local distance = param[i]*120*direction
                -- Предотвращаем возможное переполнение.
                -- Значение должно поместиться в 16 бит.
                distance = distance % 2^16
                -- Значение должно остаться кратным 120
                distance = math.floor(distance/120)*120
                modifer  = bit.bor(bit.lshift(distance, 16), modifer)
            elseif type(param[i]) == "cdata"  then
                handle   = param[i]
            end
        end
        handle = handle or window
        return x, y, modifer, keyboard_modifer, handle, 0, "ERROR_SUCCESS"
    end

    -- Конструктор комбинированных кликов down+up.
    local constructor_down_up = function(method, down, up, move_raw)
        -- Изибегаем return'а анонимной функции
        -- чтобы при отладке работала трасировка.
        local f = function(...)
            local x, y, modifer, keyboard_modifer, handle, err, verbose = param_parser(nil, mouse[method].mod_inner, ...)
            if err ~= 0 then return nil, nil, err, verbose end

            -- Сохраняем текущую позицию мыши.
            local current_x = nil
            local current_y = nil
            if mouse[method].move_after_click and mouse[method].move_before_click then
                current_x, current_y = mouse.mousepos()
            end

            -- Ждем ограничения по времени после
            -- предыдущих кликов.
            local timeout_outer = mouse[method].clock_up + mouse[method].timeout_outer
            repeat
                local clock = os.clock()
            until timeout_outer < clock
            or    sys.sleep(1)

            -- Перемещаем мышь перед кликом.
            if mouse[method].move_before_click and mouse[method].move_split then
                if mouse[method].move_func == mouse[method].move then
                    move_raw(x, y)
                else
                    mouse[method].move_func(x, y)
                end
                sys.sleep(mouse[method].timeout_move)
            end

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["down"](keyboard_modifer[i])
            end

            -- Клик
            local result_down = down(x, y, modifer, handle)
            sys.sleep(mouse[method].timeout_inner)
            local result_up   = up(x, y, modifer, handle)

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["up"](keyboard_modifer[i])
            end

            -- Возвращаем мышь
            if mouse[method].move_after_click and mouse[method].move_before_click then
                if mouse[method].move_func == mouse[method].move then
                    move_raw(current_x, current_y)
                else
                    mouse[method].move_func(current_x, current_y)
                end
            end

            mouse[method].clock_up = os.clock()
            return result_down, result_up, 0, "ERROR_SUCCESS"
        end
        return f
    end

    for method, msg_func in pairs(message_method) do
        local constructor_down_up = constructor_down_up
        mouse[method] = {}
        -- Задаем минимальное время
        -- между кликами.

        -- Таймаут между перемещением
        -- в точку клика и down.
        -- Имеет смысл только при
        -- move_split = true.
        mouse[method].timeout_move      = 0

        -- Внутренний таймаут
        -- между down и up.
        mouse[method].timeout_inner     = 10

        -- Внешний таймаут
        -- между up и down
        mouse[method].timeout_outer     = 10

        -- Объявляем время прошлого клика.
        mouse[method].clock_up          = -2^52

        -- Перемещение и клик могут быть
        -- объедены в один вызов WinAPI
        -- функции. Разделять?
        mouse[method].move_split        = true

        -- Включить перемещение мыши
        -- перед кликом.
        mouse[method].move_before_click = false

        -- Возвращать мышь после клика
        -- на предыдущую позицию.
        mouse[method].move_after_click  = false

        -- Данная функция не имеет флагов
        -- shift, ctrl, alt. Для их нажатия
        -- необходимо использовать внешнуюю
        -- фукнцию.
        mouse[method].mod_func          = kbd[method]

        -- Включить отправку модификаторов
        -- shift и ctrl в качестве флага клика.
        mouse[method].mod_inner         = false


        -- Создаем базовые функции
        -- без какого либо парсинга
        -- чтобы избежать двойной
        -- обработки параметров при
        -- кликах подразумевающих
        -- и down и up
        -- (left/right/middle/m4/m5).
        local raw = {}
        for click_name, button in pairs(click_type) do
            -- Для кнопок mouse4 и mouse5
            -- отсылается один и тот же
            -- message, а тип кнопки (4 или 5)
            -- задается флагом в wparam.
            --
            -- mouse4?
            if     click_name == "m4_down"
            or     click_name == "m4_up"
            or     click_name == "m4_double" then
                raw[click_name] = function(x, y, modifer, handle)
                    return msg_func(button, modifer + XBUTTON1, y * 0x10000 + x, handle)
                end
            -- mouse5?
            elseif click_name == "m5_down"
            or     click_name == "m5_up"
            or     click_name == "m5_double" then
                raw[click_name] = function(x, y, modifer, handle)
                    return msg_func(button, modifer + XBUTTON2, y * 0x10000 + x, handle)
                end
            else
                raw[click_name] = function(x, y, modifer, handle)
                    return msg_func(button, modifer, y * 0x10000 + x, handle)
                end
            end
        end

        -- Оборачиваем базовые функции
        -- Подставляя хэндлы,
        -- рассчитываем смещения и хэндлы
        -- для abs координат,
        -- учитываем внутренние и внешние
        -- таймауты.
        for click_name, func in pairs(raw) do
            local direction = 1
            if click_name == "wheel_down"
            or click_name == "wheel_left" then
                direction = -1
            end
            mouse[method][click_name] = function(...)
                local x, y, modifer, handle = param_parser(direction, mouse[method].mod_inner, ...)
                return func(x, y, modifer, handle)
            end
        end

        mouse[method].left   = constructor_down_up(method, raw.left_down,   raw.left_up  )
        mouse[method].right  = constructor_down_up(method, raw.right_down,  raw.right_up )
        mouse[method].middle = constructor_down_up(method, raw.middle_down, raw.middle_up)
        mouse[method].m4     = constructor_down_up(method, raw.m4_down,     raw.m4_up    )
        mouse[method].m5     = constructor_down_up(method, raw.m5_down,     raw.m5_up    )

        -- Клики игнорируют передаваемые
        -- координаты.
        -- Задаем функцию move, которая
        -- будет использована для передвижения
        -- мыши перед кликом.
        mouse[method].move_func = mouse[method].move
    end
end



-- EVENT
-- EVENT
-- EVENT
do
    if ffi.arch == "x64" then
        ffi.cdef[[
                typedef unsigned __int64 ULONG_PTR;
            ]]
    else
        ffi.cdef[[
                typedef unsigned long ULONG_PTR;
            ]]
    end
    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          void*     HWND;
        int GetSystemMetrics(int nIndex);
        void mouse_event(
            DWORD     dwFlags,
            DWORD     dx,
            DWORD     dy,
            DWORD     dwData,
            ULONG_PTR dwExtraInfo
        );
    ]]
    local SM_CXSCREEN              = 0
    local SM_CYSCREEN              = 1

    -- При работе с MOUSEEVENTF_ABSOLUTE
    -- рассчет идет по сетке от 0-65535
    -- вне зависимости от разрешения монитора.
    -- Необходимо для конвертации координат.
    local screenWidth              = user32.GetSystemMetrics(SM_CXSCREEN)
    local screenHeight             = user32.GetSystemMetrics(SM_CYSCREEN)

    -- Типы возможных базовых кликов.
    -- Все клики указываются, как флаги
    -- и могут быть суммированы.
    local click_type               = {}
    click_type.move                = 0x0001 -- MOUSEEVENTF_MOVE
    click_type.left_down           = 0x0002 -- MOUSEEVENTF_LEFTDOWN
    click_type.left_up             = 0x0004 -- MOUSEEVENTF_LEFTUP
    click_type.right_down          = 0x0008 -- MOUSEEVENTF_RIGHTDOWN
    click_type.right_up            = 0x0010 -- MOUSEEVENTF_RIGHTUP
    click_type.middle_down         = 0x0020 -- MOUSEEVENTF_MIDDLEDOWN
    click_type.middle_up           = 0x0040 -- MOUSEEVENTF_MIDDLEUP
    click_type.m4_down             = 0x0080 -- MOUSEEVENTF_XDOWN
    click_type.m4_up               = 0x0100 -- MOUSEEVENTF_XUP
    click_type.m5_down             = 0x0080 -- MOUSEEVENTF_XDOWN
    click_type.m5_up               = 0x0100 -- MOUSEEVENTF_XUP
    click_type.wheel_down          = 0x0800 -- MOUSEEVENTF_WHEEL
    click_type.wheel_up            = 0x0800 -- MOUSEEVENTF_WHEEL
    click_type.wheel_left          = 0x1000 -- MOUSEEVENTF_HWHEEL
    click_type.wheel_right         = 0x1000 -- MOUSEEVENTF_HWHEEL
    -- Следующий флаг принудительно
    -- включен для всех вызовов.
    -- Это не относительные/абсолютные
    -- координаты в пилотовском смысле.
    -- По сути это переключатель между
    -- режимом дельты координат
    -- от текущего положения курсора
    -- и включения альтернативной
    -- координатной сетки с абсолютными
    -- координатами относительно экрана.
    -- click_type.abs              = 0x8000 -- MOUSEEVENTF_ABSOLUTE

    local XBUTTON1                 = 0x0001 -- mouse4 flag
    local XBUTTON2                 = 0x0002 -- mouse5 flag
    local MOUSEEVENTF_ABSOLUTE     = 0x8000

    mouse.event = {}
    -- Задаем минимальное время
    -- между кликами.
    --
    -- Таймаут между перемещением
    -- в точку клика и down.
    -- Имеет смысл только при
    -- move_split = true.
    mouse.event.timeout_move      = 0
    --
    -- Внутренний таймаут
    -- между down и up.
    mouse.event.timeout_inner     = 10
    --
    -- Внешний таймаут
    -- между up и down
    mouse.event.timeout_outer     = 10
    --
    -- Объявляем время прошлого клика.
    mouse.event.clock_up          = -2^52

    -- Перемещение и клик могут быть
    -- объедены в один вызов WinAPI
    -- функции. Разделять?
    mouse.event.move_split        = false

    -- Включить перемещение мыши
    -- перед кликом.
    mouse.event.move_before_click = true

    -- Возвращать мышь после клика
    -- на предыдущую позицию.
    mouse.event.move_after_click  = true

    -- Данная функция не имеет флагов
    -- shift, ctrl, alt. Для их нажатия
    -- необходимо использовать внешнуюю
    -- фукнцию.
    mouse.event.mod_func          = kbd.event

    -- Парсер параметров
    local param_parser = function(direction, ...)
        local abs_flag         = nil
        local handle           = nil
        local extra_flag       = 0
        local extra_data       = 0
        local keyboard_modifer = {}
        local param            = {...}

        -- Проверяем координаты заданны массивом
        -- или переменными.
        -- start_i - стартовая позиция парсинга
        -- массива с остальными параметрами.
        -- Первые позиции обрабатываются
        -- спцифически (координаты)
        local x, y, start_i, err, verbose = parse_crds(param)
        if err ~= 0 then return nil, nil, nil, nil, err, verbose end

        for i = start_i, #param do
            if     type(param[i]) == "string" then
                -- Координаты абсолютные.
                if param[i] == "abs" then
                    abs_flag = true
                elseif param[i] == "move"
                    or param[i] == "left_down"
                    or param[i] == "left_up"
                    or param[i] == "right_down"
                    or param[i] == "right_up"
                    or param[i] == "middle_down"
                    or param[i] == "middle_up"
                    or param[i] == "wheel_up"
                    or param[i] == "wheel_right" then
                    extra_flag  =  extra_flag + click_type[param[i]]
                elseif param[i] == "wheel_down"
                    or param[i] == "wheel_left"  then
                    extra_flag  =  extra_flag + click_type[param[i]]
                    -- Две строки ниже не трогать.
                    -- Мы не знаем в каком порядке придут
                    -- модификаторы и непосредственно
                    -- количество прокруток.
                    direction   =  -1
                    extra_data  =  extra_data * direction
                elseif param[i] == "m4_down"
                    or param[i] == "m4_up"       then
                    extra_flag  =  extra_flag + click_type[param[i]]
                    extra_data  =  extra_data + XBUTTON1
                elseif param[i] == "m5_down"
                    or param[i] == "m5_up"       then
                    extra_flag  =  extra_flag + click_type[param[i]]
                    extra_data  =  extra_data + XBUTTON2
                elseif param[i] == "shift"
                    or param[i] == "ctrl"
                    or param[i] == "alt"         then
                    keyboard_modifer[#keyboard_modifer+1] = param[i]
                end
            elseif type(param[i]) == "number" then
                local distance = 0
                -- Вся прокрутка идет с шагом 120.
                -- Виндовые ограничения.
                distance = param[i]*120
                -- Предотвращаем возможное переполнение.
                -- Значение должно поместиться в 16 бит.
                distance = distance % 2^16
                -- Значение должно остаться кратным 120
                extra_data = math.floor(distance/120)*120*direction
            elseif type(param[i]) == "cdata"  then
                handle   = param[i]
            end
        end
        handle = handle or window
        if not abs_flag then
            local pos = wnd.windowpos(handle)
            x = pos.x + x
            y = pos.y + y
        end

        local xNorm = math.floor(((x+0.5)* 65535) / screenWidth)
        local yNorm = math.floor(((y+0.5)* 65535) / screenHeight)

        mouse.event.clock_up = os.clock()

        return xNorm, yNorm, extra_flag, extra_data, keyboard_modifer, 0, "ERROR_SUCCESS"
    end

    -- Конструктор комбинированных кликов down+up.
    local constructor_down_up = function(down, up, move_raw)
        -- Изибегаем return'а анонимной функции
        -- чтобы при отладке работала трасировка.
        local f = function(...)
            local x, y = parse_crds({...})
            local direction = 1
            local xNorm, yNorm, extra_flag, extra_data, keyboard_modifer, err, verbose = param_parser(direction, ...)
            if not xNorm then return nil, nil, err, verbose end

            -- Сохраняем текущую позицию мыши.
            local current_x = nil
            local current_y = nil
            if mouse.event.move_after_click and mouse.event.move_before_click then
                current_x, current_y = mouse.mousepos()
            end

            -- Нужно добавить в основной клик
            -- флаг перемещения.
            if mouse.event.move_before_click and not mouse.event.move_split then
                extra_flag = bit.bor(extra_flag, click_type.move)
            end

            -- Ждем ограничения по времени после
            -- предыдущих кликов.
            local timeout_outer = mouse.event.clock_up + mouse.event.timeout_outer
            repeat
            until timeout_outer < os.clock()
            or    sys.sleep(1)

            -- Перемещаем мышь перед кликом.
            if mouse.event.move_before_click and mouse.event.move_split then
                if mouse.event.move_func == mouse.event.move then
                    move_raw(xNorm, yNorm)
                else
                    mouse.event.move_func(x, y)
                end
                sys.sleep(mouse.event.timeout_move)
            end

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["down"](keyboard_modifer[i])
            end

            -- Клик.
            local result_down = down(xNorm, yNorm, extra_flag, extra_data)
            sys.sleep(mouse.event.timeout_inner)
            local result_up   = up(xNorm, yNorm, extra_flag, extra_data)

            -- Отпускаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["up"](keyboard_modifer[i])
            end

            -- Возвращаем мышь
            if mouse.event.move_after_click and mouse.event.move_before_click then
                mouse.event.move_func(current_x, current_y)
            end

            return result_down, result_up, 0, "ERROR_SUCCESS"
        end
        return f
    end

    -- Создаем базовые функции
    -- без какого либо парсинга
    -- чтобы избежать двойной
    -- обработки параметров при
    -- кликах подразумевающих
    -- и down и up
    -- (left/right/middle/m4/m5).
    local raw = {}
    for click_name, button in pairs(click_type) do
        -- включаем abs координаты.
        button = bit.bor(button, MOUSEEVENTF_ABSOLUTE)

        -- Для кнопок mouse4 и mouse5
        -- отсылается один и тот же
        -- флаг клика, а тип кнопки (4 или 5)
        -- задается флагом в wparam.
        -- m4, m5, wheel_* не совместимы
        -- друг с другом.
        if  click_name  == "m4_down"
            or  click_name  == "m4_up"
            or  click_name  == "m4_double" then
            raw[click_name]  = function(xNorm, yNorm, extra_flag, extra_data)
                local click  = bit.bor(button, extra_flag)
                extra_data   = bit.bor(extra_data, XBUTTON1)
                local result = user32.mouse_event(click, xNorm, yNorm, extra_data, 0)
                return result
            end

        elseif  click_name  == "m5_down"
        or      click_name  == "m5_up"
        or      click_name  == "m5_double" then
            raw[click_name]  = function(xNorm, yNorm, extra_flag, extra_data)
                local click  = bit.bor(button, extra_flag)
                extra_data   = bit.bor(extra_data, XBUTTON2)
                local result = user32.mouse_event(click, xNorm, yNorm, extra_data, 0)
                return result
            end

        -- Прочие функции.
        else
            raw[click_name]  = function(xNorm, yNorm, extra_flag, extra_data)
                local click  = bit.bor(button, extra_flag)
                local result = user32.mouse_event(click, xNorm, yNorm, extra_data, 0)
                return result
            end
        end
    end

    -- Оборачиваем базовые функции
    -- Подставляя хэндлы,
    -- рассчитываем смещения и хэндлы
    -- для abs координат,
    -- учитываем внутренние и внешние
    -- таймауты.
    for click_name, func in pairs(raw) do
        local direction = 1
        if click_name == "wheel_down"
        or click_name == "wheel_left" then
            direction = -1
        end
        mouse.event[click_name] = function(x, y, ...)
            -- Парсим параметры.
            local xNorm, yNorm, extra_flag, extra_data, keyboard_modifer, err, verbose = param_parser(direction, x, y, ...)
            if not xNorm then return nil, err, verbose end

            -- Сохраняем текущую позицию мыши.
            local current_x = nil
            local current_y = nil
            if mouse.event.move_after_click then
                current_x, current_y = mouse.mousepos()
            end

            -- Нужно добавить в основной клик
            -- флаг перемещения.
            if mouse.event.move_before_click and not mouse.event.move_split then
                extra_flag = bit.bor(extra_flag, click_type.move)
            end

            -- Перемещаем мышь перед кликом.
            if mouse.event.move_before_click and mouse.event.move_split
            and click_name ~= "move" then
                if mouse.event.move_func == mouse.event.move then
                    raw.move(xNorm, yNorm)
                else
                    mouse.event.move_func(x, y, 0, 0)
                end
            end

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["down"](keyboard_modifer[i])
            end

            -- Совершаем клик
            local result = func(xNorm, yNorm, extra_flag, extra_data, 0)

            -- Отпускаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["up"](keyboard_modifer[i])
            end

            -- Возвращаем мышь
            if mouse.event.move_after_click then
                if mouse.event.move_func == mouse.event.move then
                    raw.move(current_x, current_y, 0, 0)
                else
                    mouse.event.move_func(x, y)
                end
            end

            return result
        end
    end
    mouse.event.left   = constructor_down_up(raw.left_down,   raw.left_up,   raw.move)
    mouse.event.right  = constructor_down_up(raw.right_down,  raw.right_up,  raw.move)
    mouse.event.middle = constructor_down_up(raw.middle_down, raw.middle_up, raw.move)
    mouse.event.m4     = constructor_down_up(raw.m4_down,     raw.m4_up,     raw.move)
    mouse.event.m5     = constructor_down_up(raw.m5_down,     raw.m5_up,     raw.move)

    -- Клики игнорируют передаваемые
    -- координаты.
    -- Задаем функцию move, которая
    -- будет использована для передвижения
    -- мыши перед кликом.
    mouse.event.move_func = mouse.event.move
end



-- INPUT
-- INPUT
-- INPUT
do
    if ffi.arch == "x64" then
        ffi.cdef[[
            typedef unsigned __int64 ULONG_PTR;
        ]]
    else
        ffi.cdef[[
            typedef unsigned long ULONG_PTR;
        ]]
    end
    ffi.cdef[[
        typedef          long      LONG;
        typedef unsigned int       DWORD;
        typedef unsigned short     WORD;
        typedef unsigned int       UINT;
    ]]
    sys.define[[
        typedef struct tagMOUSEINPUT {
            LONG      dx;
            LONG      dy;
            DWORD     mouseData;
            DWORD     dwFlags;
            DWORD     time;
            ULONG_PTR dwExtraInfo;
        } MOUSEINPUT, *PMOUSEINPUT, *LPMOUSEINPUT;
    ]]
    sys.define[[
        typedef struct tagKEYBDINPUT {
            WORD      wVk;
            WORD      wScan;
            DWORD     dwFlags;
            DWORD     time;
            ULONG_PTR dwExtraInfo;
        } KEYBDINPUT, *PKEYBDINPUT, *LPKEYBDINPUT;
    ]]
    sys.define[[
        typedef struct tagHARDWAREINPUT {
            DWORD uMsg;
            WORD  wParamL;
            WORD  wParamH;
        } HARDWAREINPUT, *PHARDWAREINPUT, *LPHARDWAREINPUT;
    ]]
    sys.define[[
    typedef struct tagINPUT {
        DWORD type;
        union {
            MOUSEINPUT    mi;
            KEYBDINPUT    ki;
            HARDWAREINPUT hi;
        };
    } INPUT, *PINPUT, *LPINPUT;
    ]]
    ffi.cdef[[
        UINT SendInput(
            UINT    cInputs,
            LPINPUT pInputs,
            int     cbSize
        );
    ]]
    local INPUT_MOUSE              = 0
    --local INPUT_KEYBOARD           = 1
    --local INPUT_HARDWARE           = 2


    local action_count             = 1
    local input                    = ffi.new   ("INPUT")
          input.type               = INPUT_MOUSE
    local input_size               = ffi.sizeof( input )
    local mi                       = input.mi
          mi.time                  = 0
          mi.dwExtraInfo           = 0

    local SM_CXSCREEN              = 0
    local SM_CYSCREEN              = 1

    -- При работе с MOUSEEVENTF_ABSOLUTE
    -- рассчет идет по сетке от 0-65535
    -- вне зависимости от разрешения монитора.
    -- Необходимо для конвертации координат.
    local screenWidth              = user32.GetSystemMetrics(SM_CXSCREEN)
    local screenHeight             = user32.GetSystemMetrics(SM_CYSCREEN)

    -- Типы возможных базовых кликов.
    -- Все клики указываются, как флаги
    -- и могут быть суммированы.
    local click_type               = {}
    click_type.move                = 0x0001 -- MOUSEEVENTF_MOVE
    click_type.left_down           = 0x0002 -- MOUSEEVENTF_LEFTDOWN
    click_type.left_up             = 0x0004 -- MOUSEEVENTF_LEFTUP
    click_type.right_down          = 0x0008 -- MOUSEEVENTF_RIGHTDOWN
    click_type.right_up            = 0x0010 -- MOUSEEVENTF_RIGHTUP
    click_type.middle_down         = 0x0020 -- MOUSEEVENTF_MIDDLEDOWN
    click_type.middle_up           = 0x0040 -- MOUSEEVENTF_MIDDLEUP
    click_type.m4_down             = 0x0080 -- MOUSEEVENTF_XDOWN
    click_type.m4_up               = 0x0100 -- MOUSEEVENTF_XUP
    click_type.m5_down             = 0x0080 -- MOUSEEVENTF_XDOWN
    click_type.m5_up               = 0x0100 -- MOUSEEVENTF_XUP
    click_type.wheel_down          = 0x0800 -- MOUSEEVENTF_WHEEL
    click_type.wheel_up            = 0x0800 -- MOUSEEVENTF_WHEEL
    click_type.wheel_left          = 0x1000 -- MOUSEEVENTF_HWHEEL
    click_type.wheel_right         = 0x1000 -- MOUSEEVENTF_HWHEEL
    -- Следующий флаг не является кликом,
    -- а указывает на объединение всех
    -- флагов в один клик либо создание
    -- очереди кликов. По умолчанию - объединение.
    -- click_type.nocoal           = 0x2000 -- MOUSEEVENTF_MOVE_NOCOALESCE
    --
    -- Следующий флаг не является кликом,
    -- указывает использование абсолютных
    -- координат не экрана, а всего рабочего
    -- стола. Должен использоваться
    -- совместно с MOUSEEVENTF_ABSOLUTE.
    -- click_type.desk             = 0x4000 -- MOUSEEVENTF_VIRTUALDESK
    --
    -- Следующий флаг принудительно
    -- включен для всех вызовов.
    -- Это не относительные/абсолютные
    -- координаты в пилотовском смысле.
    -- По сути это переключатель между
    -- режимом дельты координат
    -- от текущего положения курсора
    -- и включения альтернативной
    -- координатной сетки с абсолютными
    -- координатами относительно экрана.
    -- click_type.abs              = 0x8000 -- MOUSEEVENTF_ABSOLUTE

    local MOUSEEVENTF_ABSOLUTE     = 0x8000

    local XBUTTON1                 = 0x0001 -- mouse4 flag
    local XBUTTON2                 = 0x0002 -- mouse5 flag


    mouse.input = {}
    -- Задаем минимальное время
    -- между кликами.
    --
    -- Таймаут между перемещением
    -- в точку клика и down.
    -- Имеет смысл только при
    -- move_split = true.
    mouse.input.timeout_move      = 0
    --
    -- Внутренний таймаут
    -- между down и up.
    mouse.input.timeout_inner     = 10
    --
    -- Внешний таймаут
    -- между up и down
    mouse.input.timeout_outer     = 10
    --
    -- Объявляем время прошлого клика.
    mouse.input.clock_up          = -2^52

    -- Перемещение и клик могут быть
    -- объедены в один вызов WinAPI
    -- функции. Разделять?
    mouse.input.move_split        = false

    -- Включить перемещение мыши
    -- перед кликом.
    mouse.input.move_before_click = true

    -- Возвращать мышь после клика
    -- на предыдущую позицию.
    mouse.input.move_after_click  = true

    -- Данная функция не имеет флагов
    -- shift, ctrl, alt. Для их нажатия
    -- необходимо использовать внешнуюю
    -- фукнцию.
    mouse.input.mod_func          = kbd.input

    -- Парсер параметров
    local param_parser = function(direction, ...)
        local abs_flag = nil
        local handle   = nil
        local param    = {...}

        mi.mouseData = 0
        mi.dwFlags   = 0

        -- Проверяем координаты заданны массивом
        -- или переменными.
        -- start_i - стартовая позиция парсинга
        -- массива с остальными параметрами.
        -- Первые позиции обрабатываются
        -- спцифически (координаты)
        local x, y, start_i, err, verbose = parse_crds(param)
        if err ~= 0 then return nil, nil, nil, nil, err, verbose end

        local keyboard_modifer = {}
        for i = start_i, #param do
            if     type(param[i]) == "string" then
                -- Координаты абсолютные.
                if param[i] == "abs"             then
                    abs_flag = true
                elseif param[i] == "move"
                    or param[i] == "left_down"
                    or param[i] == "left_up"
                    or param[i] == "right_down"
                    or param[i] == "right_up"
                    or param[i] == "middle_down"
                    or param[i] == "middle_up"
                    or param[i] == "wheel_down"
                    or param[i] == "wheel_left"  then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type[param[i]])
                elseif param[i] == "wheel_up"
                    or param[i] == "wheel_right" then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type[param[i]])
                    -- Две строки ниже не трогать.
                    -- Мы не знаем в каком порядке придут
                    -- модификаторы и непосредственно
                    -- количество прокруток.
                    direction   =  -1
                    mi.mouseData =  mi.mouseData * direction
                elseif param[i] == "m4_down"
                    or param[i] == "m4_up"       then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type[param[i]])
                    mi.mouseData = bit.bor(mi.mouseData, XBUTTON1)
                elseif param[i] == "m5_down"
                    or param[i] == "m5_up"       then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type[param[i]])
                    mi.mouseData = bit.bor(mi.mouseData, XBUTTON2)
                elseif param[i] == "shift"
                    or param[i] == "ctrl"
                    or param[i] == "alt"         then
                    keyboard_modifer[#keyboard_modifer+1] = param[i]
                end
            elseif type(param[i]) == "number" then
                local distance = 0
                -- Вся прокрутка идет с шагом 120.
                -- Виндовые ограничения.
                distance = param[i]*120
                -- Предотвращаем возможное переполнение.
                -- Значение должно поместиться в 16 бит.
                distance = distance % 2^16
                -- Значение должно остаться кратным 120
                distance = math.floor(distance/120)*120*direction
                mi.mouseData = bit.bor(mi.mouseData, distance)
            elseif type(param[i]) == "cdata"  then
                handle   = param[i]
            end
        end

        handle = handle or window
        if not abs_flag then
            local pos = wnd.windowpos(handle)
            x = pos.x + x
            y = pos.y + y
        end

        mi.dx = math.floor(((x+0.5)* 65535) / screenWidth)
        mi.dy = math.floor(((y+0.5)* 65535) / screenHeight)

        return keyboard_modifer, 0, "ERROR_SUCCESS"
    end

    -- Конструктор комбинированных кликов down+up.
    local constructor_down_up = function(down, up, move_raw)
        -- Изибегаем return'а анонимной функции
        -- чтобы при отладке работала трасировка.
        local f = function(...)
            local x, y = parse_crds({...})
            local keyboard_modifer, err, verbose = param_parser(1, ...)
            if err ~= 0 then return nil, nil, err, verbose end

            -- Сохраняем текущую позицию мыши.
            local current_x = nil
            local current_y = nil
            if mouse.input.move_after_click and mouse.input.move_before_click then
                current_x, current_y = mouse.mousepos()
            end

            -- Нужно добавить в основной клик
            -- флаг перемещения.
            local move_add = mouse.input.move_before_click and not mouse.input.move_split

            -- Ждем ограничения по времени после
            -- предыдущих кликов.
            local timeout_outer = mouse.input.clock_up + mouse.input.timeout_outer
            repeat
            until timeout_outer < os.clock()
            or    sys.sleep(1)

            -- Перемещаем мышь перед кликом.
            if mouse.input.move_before_click and mouse.input.move_split then
                if mouse.input.move_func == mouse.input.move then
                    move_raw(x, y)
                else
                    mouse.input.move_func(x, y)
                end
                sys.sleep(mouse.input.timeout_move)
            end

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["down"](keyboard_modifer[i])
            end

            local result_down = down(move_add)
            sys.sleep(mouse.input.timeout_inner)
            local result_up   = up(move_add)

            -- Отжимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["up"](keyboard_modifer[i])
            end

            -- Возвращаем мышь
            if mouse.input.move_after_click and mouse.input.move_before_click then
                if mouse.input.move_func == mouse.input.move then
                    move_raw(current_x, current_y)
                else
                    mouse.input.move_func(current_x, current_y)
                end
            end

            mouse.input.clock_up = os.clock()
            return result_down, result_up, 0, "ERROR_SUCCESS"
        end
        return f
    end

    -- Создаем базовые функции
    -- без какого либо парсинга
    -- чтобы избежать двойной
    -- обработки параметров при
    -- кликах подразумевающих
    -- и down и up
    -- (left/right/middle/m4/m5).
    local raw = {}
    for click_name, button in pairs(click_type) do
        button = bit.bor(button, MOUSEEVENTF_ABSOLUTE)
        -- Для кнопок mouse4 и mouse5
        -- отсылается один и тот же
        -- флаг клика, а тип кнопки (4 или 5)
        -- задается флагом в wparam.
        -- m4, m5, wheel_* не совместимы
        -- друг с другом.
        if  click_name == "m4_down"
            or  click_name == "m4_up"
            or  click_name == "m4_double" then
            raw[click_name] = function(add_move)
                mi.dwFlags = bit.bor(button, mi.dwFlags)
                if add_move then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type.move)
                end
                mi.mouseData = bit.bor(mi.mouseData, XBUTTON1)
                local result = user32.SendInput(action_count, input, input_size)
                return result
            end

        elseif  click_name == "m5_down"
        or      click_name == "m5_up"
        or      click_name == "m5_double" then
            raw[click_name] = function(add_move)
                mi.dwFlags = bit.bor(button, mi.dwFlags)
                if add_move then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type.move)
                end
                mi.mouseData = bit.bor(mi.mouseData, XBUTTON2)
                local result = user32.SendInput(action_count, input, input_size)
                return result
            end

        -- Прочие функции.
        else
            raw[click_name] = function(add_move)
                mi.dwFlags = bit.bor(button, mi.dwFlags)
                if add_move then
                    mi.dwFlags = bit.bor(mi.dwFlags, click_type.move)
                end
                local result = user32.SendInput(action_count, input, input_size)
                return result
            end
        end
    end

    -- Оборачиваем базовые функции
    -- Подставляя хэндлы,
    -- рассчитываем смещения и хэндлы
    -- для abs координат,
    -- учитываем внутренние и внешние
    -- таймауты.
    for click_name, func in pairs(raw) do
        local direction = 1
        if click_name == "wheel_down"
        or click_name == "wheel_left" then
            direction = -1
        end

        mouse.input[click_name] = function(x, y, ...)
            -- Нужно добавить в основной клик
            -- флаг перемещения.
            local add_move = mouse.input.move_before_click and not mouse.input.move_split

            -- Парсим параметры.
            local keyboard_modifer, err, verbose = param_parser(direction, x, y, ...)
            if err ~= 0 then return nil, err, verbose end

            -- Сохраняем текущую позицию мыши.
            local current_x = nil
            local current_y = nil
            if mouse.input.move_after_click then
                current_x, current_y = mouse.mousepos()
            end

            -- Перемещаем мышь перед кликом.
            if mouse.input.move_before_click and mouse.input.move_split
            and click_name ~= "move" then
                if mouse.input.move_func == mouse.input.move then
                    raw.move(x, y)
                else
                    mouse.input.move_func(x, y)
                end
            end

            -- Зажимаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["down"](keyboard_modifer[i])
            end

            -- Совершаем клик.
            local result = func(add_move)

            -- Отпускаем shift, ctrl, alt.
            for i = 1, #keyboard_modifer do
                mouse.input.mod_func["up"](keyboard_modifer[i])
            end

            -- Возвращаем мышь
            if mouse.input.move_after_click and click_name ~= "move" then
                if mouse.input.move_func == mouse.input.move then
                    raw.move(current_x, current_y)
                else
                    mouse.input.move_func(current_x, current_y)
                end
            end

            return result
        end
    end
    mouse.input.left   = constructor_down_up(raw.left_down,   raw.left_up  , raw.move)
    mouse.input.right  = constructor_down_up(raw.right_down,  raw.right_up , raw.move)
    mouse.input.middle = constructor_down_up(raw.middle_down, raw.middle_up, raw.move)
    mouse.input.m4     = constructor_down_up(raw.m4_down,     raw.m4_up    , raw.move)
    mouse.input.m5     = constructor_down_up(raw.m5_down,     raw.m5_up    , raw.move)

    -- Клики игнорируют передаваемые
    -- координаты.
    -- Задаем функцию move, которая
    -- будет использована для передвижения
    -- мыши перед кликом.
    mouse.input.move_func = mouse.input.move
end

return mouse








