--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]



local ffi           = require[[ffi]]
local sys           = require[[lua_system\system]]

local kernel32      = ffi.load("kernel32")
local winmm         = ffi.load("Winmm")

local time = {}


-- sleep(time_to_sleep)
-- приостановить работу.
--
-- time_to_sleep - время приостановки в секундах.
--                 Точность до 1/1000 секунды.
do

    ffi.cdef[[
        typedef unsigned int       UINT;
        typedef          UINT      MMRESULT;

        void Sleep(unsigned int dwMilliseconds);
        MMRESULT timeBeginPeriod(
            UINT uPeriod
        );
        MMRESULT timeEndPeriod(
            UINT uPeriod
        );
    ]]

    time.sleep = function(time_to_sleep)
        winmm.timeBeginPeriod(1)
        kernel32.Sleep(time_to_sleep)
        winmm.timeEndPeriod(1)
    end
end


-- result = sec()
-- Получает значение секунд текущего времени.
--
-- result - секунды.
time.sec   = function()
    return tonumber(os.date("%S"))
end


-- result = min()
-- Получает значение минут текущего времени.
--
-- result - минуты.
time.min   = function()
    return tonumber(os.date("%M"))
end


-- result = hour()
-- Получает значение часов текущего времени.
--
-- result - часы.
time.hour  = function()
    return tonumber(os.date("%H"))
end


-- result = day()
-- Получает значение текущего дня месяца.
--
-- result - номер дня.
time.day   = function()
    return tonumber(os.date("%d"))
end


-- result = month()
-- Получает значение текущего месяца.
--
-- result - месяц.
time.month = function()
    return tonumber(os.date("%m"))
end


-- result = year()
-- Получает значение текущего года.
--
-- result - год.
time.year  = function()
    return tonumber(os.date("%Y"))
end


-- result = weekday()
-- Получает значение текущего дня недели.
--
-- result - день недели.
--          1 - понедельник
--          ...
--          7 - воскресенье
time.weekday = function()
    local day = tonumber(os.date("%w")) + 1
    if day == 8 then
        day = day - 1
    end
    return day
end


-- result = timestamp([opt_1, [opt_2]])
-- Возвращает форматированный таймштамп вида:
-- 2024.07.28 13-46-26
-- 2024.07.28 13-46-26.356 (если вызван с опцией "ms")
--
-- opt_* - опции вывода таймштампа:
--
--         ms     - выводить с тысячными долями секунды
--
--         unique - таймштамп должен отличаться от
--                  предыдущего вызова функции с
--                  включенной опцией unique.
--                  В случае, если время еще не изменилось
--                  функция встанет на ожидание и будет
--                  проверять время раз в миллисекунду.
do
    ffi.cdef[[
        typedef          short     WORD;
    ]]
    sys.define[[
        typedef struct {
            WORD wYear;
            WORD wMonth;
            WORD wDayOfWeek;
            WORD wDay;
            WORD wHour;
            WORD wMinute;
            WORD wSecond;
            WORD wMilliseconds;
        } SYSTEMTIME;
    ]]
    ffi.cdef[[
        void GetSystemTime(SYSTEMTIME *lpSystemTime);
    ]]
    local systemtime = ffi.new("SYSTEMTIME")
    local prev_stamp = ""

    time.timestamp = function(...)
        local param = {...}
        for i = 1, #param do
            param[param[i]] = true
        end

        if param.ms then
            local clock = os.clock()
            kernel32.GetSystemTime(systemtime)
            local result = os.date("%Y.%m.%d %H-%M-%S."
                                 ..string.format("%03d", systemtime.wMilliseconds))

            if param.unique then
                while result == prev_stamp do
                    time.sleep(1)
                    kernel32.GetSystemTime(systemtime)
                    result = os.date("%Y.%m.%d %H-%M-%S."
                                 ..string.format("%03d", systemtime.wMilliseconds))
                end
                prev_stamp = result
            end

            return result
        else
            local result = os.date("%Y.%m.%d %H-%M-%S")

            if param.unique then
                while result == prev_stamp do
                    time.sleep(1)
                    kernel32.GetSystemTime(systemtime)
                    result = os.date("%Y.%m.%d %H-%M-%S")
                end
                prev_stamp = result
            end

            return result
        end
    end
end


 -- ticks = clock_hp()
 -- Высокоточный таймер.
 -- Его значение выражается в количестве
 -- тактов процессора, а не фактического времени.
 --
 -- ticks - количество тактов.
 --
 -- Внимание! В связи с особенностями типов
 -- данных в luajit невозможно хранить
 -- количество тиков в стандратном number.
 -- number позволяет без потери точности
 -- использовать целые числа до 52 бит длиной,
 -- количество тиков 64 битное значение.
 -- Теоретически 52 бит при тактовой частоте
 -- в 4ГГц хватит на 35 лет, но
 -- работа данного счетчика не имеет строгих
 -- правил и сильно зависит от реализации
 -- на уровне железа из-за чего на некоторых
 -- платформах время до преполнения может
 -- уменьшится в тысячи раз. При этом значение
 -- станет не целым, а с плавающей запятой,
 -- точность будет снижена.
 -- Если требуется гарантированное выполнение
 -- с полной точностью - см. функцию clock_hp_s()
 -- Для того чтобы узнать количество тиков
 -- в секунду используйте clock_hp_resolution().
do
    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          long      LONG;
    ]]
    sys.define[[
        typedef struct {
            DWORD LowPart;
            LONG  HighPart;
        } LARGE_INTEGER;

        void QueryPerformanceCounter(LARGE_INTEGER *lpPerformanceCount);
    ]]
    local counter                 = ffi.new ("LARGE_INTEGER")

    time.clock_hp = function()
        kernel32.QueryPerformanceCounter(counter)
        return counter.HighPart * 0x100000000 + counter.LowPart
    end
end


 -- clock_hp_int64()
 -- Высокоточный таймер.
 -- Его значение выражается в количестве
 -- тактов процессора, а не фактического времени.
 --
 -- Результат работы функции будет содержаться в
 -- clock_hp_result[0] тип данных int64_t.
 -- Данное решение обусловлено с высокими
 -- накладными расходами при return int64_t.
 -- Обратите внимание, что операция:
 -- local t = clock_hp_result[0]
 -- вызовет существенные накладные расходы.
 -- Рекомендуется создать int64_t до
 -- вызова clock_hp_s и схоранять копию
 -- значения туда (при необходимости).
 -- my_result = ffi.new ("uint64_t[1]")
 -- clock_hp_int64()
 -- my_result[0] = clock_hp_result[0]
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          long      LONG;
    ]]
    sys.define[[
        typedef struct {
            DWORD LowPart;
            LONG  HighPart;
        } LARGE_INTEGER;

        BOOL QueryPerformanceCounter(LARGE_INTEGER *lpPerformanceCount);
    ]]
    local counter                 = ffi.new ("LARGE_INTEGER")
    local high_ptr                = ffi.cast("int8_t*", counter) + ffi.offsetof("LARGE_INTEGER", "HighPart")
    local low_ptr                 = ffi.cast("int8_t*", counter) + ffi.offsetof("LARGE_INTEGER", "LowPart")
    time.clock_hp_result          = ffi.new ("uint64_t[1]")
    local uint64_counter_low_ptr  = ffi.cast("uint8_t*", time.clock_hp_result)
    local uint64_counter_high_ptr = ffi.cast("uint8_t*", time.clock_hp_result) + 4

    time.clock_hp_int64 = function()
        kernel32.QueryPerformanceCounter(counter)
        ffi.copy(uint64_counter_low_ptr,  low_ptr,  4)
        ffi.copy(uint64_counter_high_ptr, high_ptr, 4)
    end
end


-- result = clock_hp_resolution()
-- Получает количество тиков в секунду при
-- использовании clock_hp() и clock_hp_s()
--
-- result - количество тиков.
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          long      LONG;
    ]]
    sys.define[[
        typedef struct {
            DWORD LowPart;
            LONG  HighPart;
        } LARGE_INTEGER;

        BOOL QueryPerformanceFrequency(LARGE_INTEGER *lpPerformanceCount);
    ]]
    local per_second              = ffi.new ("LARGE_INTEGER")

    time.clock_hp_resolution = function()
        kernel32.QueryPerformanceFrequency(per_second)
        return per_second.HighPart * 0x100000000 + per_second.LowPart
    end
end


-- sleep_hp(time)
-- Задержка с высокой точностью.
--
-- time - время задержки в секундах.
do
    local resolution  = time.clock_hp_resolution()
    time.sleep_hp = function(time_to_sleep)
        time.clock_hp_int64()
        local time_to_end = time.clock_hp_result[0] + time_to_sleep * resolution
        local i = 0
        repeat time.clock_hp_int64() i = i + 1 until time_to_end < time.clock_hp_result[0]
    end
end

return time































