--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]

local ffi           = require[[ffi]]

local sys           = {}

-- ispilot
-- Проверка в пилоте мы или нет.
-- Это переменная, а не функция.
do
    sys.ispilot = relativeaddress2absolute ~= nil or false
end

local system_error  = sys.ispilot and require[[lua_system\system_error_1251]]
                                   or require[[lua_system\system_error]]

local kernel32      = ffi.load("kernel32")
local user32        = ffi.load("user32"  )
local msvcrt        = ffi.load("msvcrt")




-- lg(obj_1, obj_2, ..., obj_n)
-- Вывод содержимого таблицы в лог.
-- Допускается вывод
-- других типов данных.
--
-- obj_* - данные для вывода в лог.
do
    sys.lg = function(...)
        local param = {...}

        for i = 1, #param do
            local data = param[i]
            local tab = ""
            local t = {} -- массив с таблицами, чтобы не уйти в рекурсию.
            local deep = 0

            local function show(data)
                -- Пишем в лог комментарий.
                deep = deep + 1
                if deep > 1 then
                    t[_G] = true
                end

                if type(data) == "table"    then
                    local element_counter = 0
                    if not t[data] then
                        t[data] = true
                        for k,v in pairs(data) do
                            element_counter = element_counter + 1
                            if  type (v) == "table" then
                                print(tab..'table: '..k)
                                tab = tab .. "    "
                                show(v)
                                tab = string.sub(tab, 1, -5)
                            else
                                if     type(v) == "nil"       then v = "nil"
                                elseif type(v) == "string"    then v = '"' .. v .. '"'
                                elseif type(v) == "number"    then v = tostring(v)
                                elseif type(v) == "function"  then v = tostring(v)
                                elseif type(v) == "thread"    then v = tostring(v)
                                elseif type(v) == "userdata"  then v = tostring(v)
                                elseif type(v) == "boolean"   then v = tostring(v)
                                elseif type(v) == "cdata"     then v = tostring(v)
                                elseif type(v) == "table"     then
                                    print(tab..""..k.." = "..v or "nil")
                                else
                                    v = "unknown data type"
                                end
                                print(tab..""..k.." = " .. v)
                            end
                        end
                        print(tab.."".."Elements in table: " .. element_counter)
                    else
                        if data == _G then
                            print(tab.."".."Table is _G")
                        else
                            print(tab.."".."Table already enumerated: " .. tostring(data))
                        end
                    end
                else
                    print('table is "' .. type(data) .. '" data type. Value: ' .. tostring(data))
                end

                --tab = ""
                --deep = 0
            end

            show(data)
        end
    end
end


-- define(str)
-- Выполняет защищенный вызов
-- ffi.cdef чтобы избежать
-- redefine на струтурах.
--
-- str - строка для define.
do
    sys.define = function(str)
        return pcall(ffi.cdef, str)
    end
end


-- err, verbose = geterr()
-- Получение кода последней ошибки
-- и его расшифровки.
--
-- err     - код ошибки
--
-- verbose - текстовая расшифровка ошибки
do
    ffi.cdef[[
        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    sys.geterr = function()
        local err = kernel32.GetLastError()
        return err, system_error[err]
    end
end


-- utf16buf, utf16len, err, verbose = utf8_to_utf16(utf8str)
-- преобразовывает строку utf-8 в utf-16 LE
--
-- utf16buf - указатель на строку в utf-16 LE
--
-- utf16len - размер utf-16 LE строки в wchar_t.
--            Для преобразования размера в байты
--            можете воспользоваться следующим способом:
--            utf16len * ffi.sizeof("wchar_t")
--
-- utf8str  - строка в utf-8.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
do
    ffi.cdef[[
        typedef unsigned int       UINT;
        typedef unsigned int       DWORD;
        typedef const    char*     LPCCH;
        typedef          wchar_t   WCHAR;
        typedef          WCHAR*    LPWSTR;
        typedef const    wchar_t*  LPCWCH;

        int MultiByteToWideChar(
            UINT   CodePage,
            DWORD  dwFlags,
            LPCCH  lpMultiByteStr,
            int    cbMultiByte,
            LPWSTR lpWideCharStr,
            int    cchWideChar
        );
    ]]
    -- Кодовая страница UTF-8
    local CP_UTF8              = 65001
    -- Флаг для обработки недопустимых символов
    local MB_ERR_INVALID_CHARS = 0x00000008

    local buffer_size = 1024
    local utf16buf = ffi.new("wchar_t[?]", buffer_size)

    sys.utf8_to_utf16 = function(utf8str)
        if not utf8str then
            return nil, 0, 0, "ERROR_SUCCESS"
        end
        if #utf8str == 0 then
            utf16buf[0] = 0
            return utf16buf, 0, 0, "ERROR_SUCCESS"
        end

        -- Рассчитываем необходимую длину UTF-16 строки в символах (wchar_t)
        local utf16len = kernel32.MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS,
                                                    utf8str, #utf8str, nil, 0)
        if utf16len == 0 then
            return nil, sys.geterr()
        end

        -- Увеличиваем буфер, если необходимо
        if utf16len > buffer_size then
            buffer_size = 2^math.ceil(math.log(utf16len, 2))
            utf16buf = ffi.new("wchar_t[?]", buffer_size)
        end

        -- Преобразуем UTF-8 в UTF-16
        local result = kernel32.MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS,
                                                  utf8str, #utf8str, utf16buf, utf16len)
        if result == 0 then
            return nil, sys.geterr()
        end

        return utf16buf, utf16len, 0, "ERROR_SUCCESS"
    end
end


-- ascii_str, err, verbose = utf16_to_ascii(utf16str, [utf16len])
-- преобразовывает строку utf-8 в ascii
--
-- ascii_str - строка в ascii
--
-- utf16str  - указатель на строку в utf-16 LE.
--
-- utf16len  - размер utf-16 LE строки в wchar_t.
--             Данный парметр является опциональным,
--             если размер не указан, то он будет
--             посчитан автоматически, однако
--             прямое указание размера ускоряет работу.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
do
    ffi.cdef[[
        typedef unsigned int       UINT;
        typedef unsigned int       DWORD;
        typedef const    wchar_t*  LPCWCH;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;
        typedef          int       BOOL;
        typedef const    char*     LPCCH;
        typedef          BOOL*     LPBOOL;

        int WideCharToMultiByte(
            UINT   CodePage,
            DWORD  dwFlags,
            LPCWCH lpWideCharStr,
            int    cchWideChar,
            LPSTR  lpMultiByteStr,
            int    cbMultiByte,
            LPCCH  lpDefaultChar,
            LPBOOL lpUsedDefaultChar
        );
    ]]

    -- Кодовая страница для ASCII
    local CP_ACP               = 0

    -- Флаги для функции WideCharToMultiByte
    -- Выбивать ошибку, если недопустимый символ.
    local MB_ERR_INVALID_CHARS = 0x00000008.

    -- Выделяем буфер для многобайтовой строки
    local buffer_size          = 1024
    local buffer               = ffi.new("char[?]", buffer_size)

    sys.utf16_to_ascii = function(utf16str, utf16len)
        if not utf16str then
            return nil, 0, "ERROR_SUCCESS"
        end

        -- Длина строки в символах (не включая завершающий нуль)
        utf16len = utf16len or ffi.C.wcslen(utf16str)
        if utf16len == 0 then
             return "", 0, "ERROR_SUCCESS"
        end

        -- Определяем необходимый размер буфера для многобайтовой строки
        local ascii_lenght = ffi.C.WideCharToMultiByte(CP_ACP,
                                                       0,
                                                       utf16str, utf16len,
                                                       nil, 0, nil, nil)
        if ascii_lenght == 0 then
            return nil, sys.geterr()
        end

        -- Увеличиваем буфер, если необходимо
        if ascii_lenght > buffer_size then
            buffer_size = 2^math.ceil(math.log(ascii_lenght, 2))
            buffer = ffi.new("char[?]", buffer_size)
        end

        -- Преобразуем UTF-16LE в ASCII
        local result = ffi.C.WideCharToMultiByte(CP_ACP, 0,
                                                 utf16str, utf16len,
                                                 buffer, buffer_size, nil, nil)
        if result == 0 then
            return nil, sys.geterr()
        end

        return ffi.string(buffer), 0, "ERROR_SUCCESS"
    end
end


-- utf16_str, err, verbose = utf8_to_ascii(utf8str)
-- преобразовывает строку utf-8 в ascii
--
-- utf16_str - строка в ascii
--
-- err       - код ошибки. 0 - успех.
--
-- verbose   - текстовая расшифровка ошибки
--
-- utf8str   - строка в uft-8.
do
    sys.utf8_to_ascii = function(utf8str)
        if not utf8str then
            return nil, 0, 0, "ERROR_SUCCESS"
        end
        local utf16, utf16len, err, verbose = sys.utf8_to_utf16(utf8str)
        if not utf16 then return nil, err, verbose end

        return sys.utf16_to_ascii(utf16, utf16len)
    end
end


-- success, err, verbose = utf16_to_file(uft16str, utf16len, file_path)
-- Сохраняет uft-16 LE строку в файл.
--
-- success   - флаг успеха выполнения. true/nil
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- utf16str  - указатель на строку в utf-16 LE.
--
-- utf16len  - размер utf-16 LE строки в wchar_t.
--             Данный парметр является опциональным,
--             если размер не указан, то он будет
--             посчитан автоматически, однако
--             прямое указание размера ускоряет работу.
do
    ffi.cdef[[
        size_t wcslen(const wchar_t *s);
    ]]

    -- Байты BOM для UTF-16LE
    local BOM_UTF16LE = "\xFF\xFE"

    sys.utf16_to_file = function(uft16str, utf16len, file_path)
        if not file_path then
            file_path = utf16len
            utf16len = ffi.C.wcslen(uft16str)
        end

        local f = io.open(file_path, "wb")
        if not f then
            return nil, sys.geterr()
        end

        -- Записываем BOM для UTF-16LE
        f:write(BOM_UTF16LE)

        -- Записываем данные UTF-16LE
        local bytes_to_write = utf16len * ffi.sizeof("wchar_t")
        f:write(ffi.string(uft16str, bytes_to_write))
        f:close()

        return true, 0, "ERROR_SUCCESS"
    end
end


-- key_value, data_size, err, verbose = registryread(fullpath)
-- Получает значение ключа реестра.
--
-- key_value - значение ключа.
--             Строки будут приведены к
--             ascii, в том числе REG_LINK
--             который хранится всегда в
--             юникоде.
--
-- data_size - размер полученных данных.
--             Актуально для binary, none,
--             string данных. При типе данных
--             REG_LINK вернет размер юникод
--             строки, но сама строка будет
--             в ascii. Т.е. этот параметр
--             не равен размеру возвращаемой
--             строки.
--
-- err       - код ошибки. 0 - успех.
--
-- verbose   - текстовая расшифровка ошибки
do
    ffi.cdef[[
        // самодуры из microsoft которые
        // переобъявляют все типы включая
        // стандартные, в чередной раз
        // решили что-то переобъявить и
        // забить на какую либо документацию.
        // Чем на самом деле является
        // LSTATUS и REGSAM они решили оставить
        // в тайне. Есть сведения не заслуживающие
        // доверия, попытался что-то изобразить.
        typedef          int       LSTATUS;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE    HKEY;
        typedef          char      CHAR;
        typedef const    CHAR*     LPCSTR;
        typedef unsigned long      DWORD;
        typedef          DWORD     REGSAM;
        typedef          HKEY*     PHKEY;
        typedef          DWORD*    LPDWORD;
        typedef unsigned char      BYTE;
        typedef          BYTE*     LPBYTE;

        LSTATUS RegOpenKeyExA(
            HKEY   hKey,
            LPCSTR lpSubKey,
            DWORD  ulOptions,
            REGSAM samDesired,
            PHKEY  phkResult
        );

        LSTATUS RegQueryValueExA(
            HKEY    hKey,
            LPCSTR  lpValueName,
            LPDWORD lpReserved,
            LPDWORD lpType,
            LPBYTE  lpData,
            LPDWORD lpcbData
        );
        LSTATUS RegCloseKey(
            HKEY hKey
        );
    ]]

    local root = {
        HKEY_CLASSES_ROOT         = ffi.cast("HKEY", 0x80000000),
        HKEY_CURRENT_USER         = ffi.cast("HKEY", 0x80000001),
        HKEY_LOCAL_MACHINE        = ffi.cast("HKEY", 0x80000002),
        HKEY_USERS                = ffi.cast("HKEY", 0x80000003),
        HKEY_PERFORMANCE_DATA     = ffi.cast("HKEY", 0x80000004),
        HKEY_CURRENT_CONFIG       = ffi.cast("HKEY", 0x80000005),
    }

    local KEY_READ                = 0x20019
    local ERROR_MORE_DATA         = 0x00EA
    local NULL_POINTER            = ffi.cast("void*", 0)

    local REG_NONE                = 0x0
    local REG_SZ                  = 0x1
    local REG_EXPAND_SZ           = 0x2
    local REG_BINARY              = 0x3
    local REG_DWORD               = 0x4
    local REG_DWORD_LITTLE_ENDIAN = 0x4
    local REG_DWORD_BIG_ENDIAN    = 0x5
    local REG_LINK                = 0x6
    local REG_MULTI_SZ            = 0x7
    local REG_QWORD               = 0xB
    local REG_QWORD_LITTLE_ENDIAN = 0xB

    local hKey                    = ffi.new("HKEY [1]")
    local lpType                  = ffi.new("DWORD[1]")
    local buffer_size             = 256
    local buffer                  = ffi.new("char [?]", buffer_size)
    local data_size               = ffi.new("DWORD[1]", 0)
    local result_64               = ffi.new("uint64_t[1]")

    sys.registryread = function(fullpath)
        -- получаем ветку реестра, путь, имя ключа.
        local root_name, subkey, valueName = fullpath:match("^([^\\]+)\\(.*)\\([^\\]+)$")

        local result = kernel32.RegOpenKeyExA(root[root_name], subkey, 0, KEY_READ, hKey)
        if result ~= 0 then
            return nil, result, system_error[result]
        end

        -- Получаем размер данных.
        result = kernel32.RegQueryValueExA(hKey[0],      valueName,
                                           nil,          lpType,
                                           NULL_POINTER, data_size)
        if result ~= 0 then
            return nil, result, system_error[result]
        end

        -- Увеличиваем буфер, если необходимо
        if data_size[0]+1 > buffer_size then
            buffer_size = 2^math.ceil(math.log(data_size[0]+1, 2))
            buffer = ffi.new("char[?]", buffer_size)
        end

        -- Считываем значение
        result = kernel32.RegQueryValueExA(hKey[0], valueName, nil, lpType, buffer, data_size)
        if result ~= 0 then
            return nil, result, system_error[result]
        end

        -- Приводим значение к луашным типам.
        -- Unicode строка будет в ascii преобразована.
        local key_value = nil
        if     lpType[0] == REG_BINARY
            or lpType[0] == REG_NONE then
            key_value = buffer

        elseif lpType[0] == REG_DWORD
            or lpType[0] == REG_DWORD_LITTLE_ENDIAN
            or lpType[0] == REG_DWORD_BIG_ENDIAN then
            key_value = buffer[0] +
                        bit.lshift(buffer[1], 8) +
                        bit.lshift(buffer[2], 16) +
                        bit.lshift(buffer[3], 24)

        elseif lpType[0] == REG_QWORD
            or lpType[0] == REG_QWORD_LITTLE_ENDIAN then
            ffi.copy(result_64, buffer, 8)
            key_value = result_64[0]

        elseif lpType[0] == REG_SZ
            or lpType[0] == REG_EXPAND_SZ
            or lpType[0] == REG_MULTI_SZ then
            -- Типы данных нуль детерминированная строка.
            -- Но функция читающая эти данные не
            -- гарантирует наличие 0x00 в конце полученной строки.
            if buffer[data_size[0]] == 0 then
                key_value = ffi.string(buffer, data_size[0]-1)
            else
                key_value = ffi.string(buffer, data_size[0])
            end

        elseif lpType[0] == REG_LINK then
            data_size[0] = sys.utf16_to_ascii(buffer, data_size[0])
        end

        result = kernel32.RegCloseKey(hKey[0])
        if result ~= 0 then
            return nil, result, system_error[result]
        end

        return key_value, data_size[0], 0, "ERROR_SUCCESS"
    end
end


-- index = indexof(tbl, val)
-- Получает номер индекса в массиве
-- по значению элемента.
-- Возвращает первый совпавший индекс.
--
-- index - найденный номер индекса.
--
-- tbl   - таблица в которой происходит поиск.
--
-- val   - значение, которое ищется в таблице.
do
    sys.indexof = function(tbl, val)
        for i = 1, #tbl do
            if tbl[i] == val then
                return i
            end
        end
        return nil
    end
end


-- result = workdirectory(dir)
-- Получает текущую рабочую директорию
--
-- result - рабочая директория.
--
-- dir    - установить новую рабочую директорию.
--          Если не задана, то просто вернет текущую.
do
    ffi.cdef[[
        int _chdir(const char* dirname);
        char *_getcwd(
            char *buffer,
            int maxlen
        );
    ]]
    local buffer_size = 1
    local buffer = ffi.new("char[?]" , buffer_size)
    local cnullp = ffi.new("char*", ffi.cast("void*", 0))

    sys.workdirectory = function(dir)
        if dir then
            local result = msvcrt._chdir(dir)
        end

        local result = msvcrt._getcwd(buffer, buffer_size)
        -- Увеличиваем буфер, если необходимо
        while result == cnullp do
            buffer_size = 2^math.ceil(math.log(buffer_size+1, 2))
            buffer = ffi.new("char[?]" , buffer_size)
            result = ffi.C._getcwd(buffer, buffer_size)
        end
        return ffi.string(buffer)
    end
end


-- result = loadlines(<path>, [empty_lines, [empty_tail]])
-- Загружает файл построчно в массив.
--
-- path        - пусть к файлу.
--
-- empty_lines - загружать пустые строки true/false.
--               По умолчанию true.
-- erease_tail - срезать загрузку пустыъ строк в хвосте файла.
--               Нужно понимать, что при выключенном
--               empty_lines пустой хвост так же загружен не будет.
--               По умолчанию false.
do
    sys.loadlines = function(path, empty_lines, erease_tail)
        empty_lines = empty_lines == nil and true or false

        local f = io.open(path, "r")
        if f then
            local result = {}
            if empty_lines then
                for line in f:lines() do
                    result[#result + 1] = line
                end
            else
                for line in f:lines() do
                    if  line ~= "" then
                        result[#result + 1] = line
                    end
                end
            end

            if erease_tail and not empty_lines then
                for i = #result, 1, -1 do
                    if result[i] == "" then
                        result[i] = nil
                    else
                        break
                    end
                end
            end
            return result
        end
    end
end

return sys



























