--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]

local ffi           = require[[ffi]]
local sys           = require[[lua_system\system]]
local system_error  = sys.ispilot and require[[lua_system\system_error_1251]]
                                   or require[[lua_system\system_error]]

local kernel32      = ffi.load("kernel32")
local user32        = ffi.load("user32"  )

local wnd           = {}











--     XXX\       /XXXX\       /XXX     XXXX      XXXX     XXXXXXXXXXX
--     \XXX       XXXXXX       XXX/     XXXXX     XXXX     XXXXXXXXXXXXX
--      XXX\     /XXXXXX\     /XXX      XXXXXX    XXXX     XXXX     XXXXX
--      \XXX     XXX/\XXX     XXX/      XXXXXXX   XXXX     XXXX      XXXX
--       XXX\   /XXX  XXX\   /XXX       XXXXXXXX  XXXX     XXXX      XXXX
--       \XXX   XXX/  \XXX   XXX/       XXXX XXXX XXXX     XXXX      XXXX
--        XXX\ /XXX    XXX\ /XXX        XXXX  XXXXXXXX     XXXX      XXXX
--        \XXX XXX/    \XXX XXX/        XXXX   XXXXXXX     XXXX      XXXX
--         XXXVXXX      XXXVXXX         XXXX    XXXXXX     XXXX     XXXXX
--         \XXXXX/      \XXXXX/         XXXX     XXXXX     XXXXXXXXXXXXX
--          XXXXX        XXXXX          XXXX      XXXX     XXXXXXXXXXX











-- define(str)
-- Выполняет защищенный вызов
-- ffi.cdef чтобы избежать
-- redefine на струтурах.
--
-- str - строка для define.
local define = function(str)
    return pcall(
                    function()
                        ffi.cdef(str)
                    end
                )
end


-- err, verbose = geterr()
-- Получение кода последней ошибки
-- и его расшифровки.
--
-- err     - код ошибки
--
-- verbose - текстовая расшифровка ошибки
local geterr = function() end
do
    ffi.cdef[[
        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    geterr = function()
        local err = kernel32.GetLastError()
        return err, system_error[err]
    end
end


-- HWND = tohandle()
-- Переобразут хэндл заданный number,
-- в HWND (он же void*).
--
-- HWND - number преобразованный в HWND.
do
    ffi.cdef[[
        typedef          void*     HWND;
    ]]
    wnd.tohandle = function(handle_number)
        return ffi.cast("HWND", handle_number)
    end
end


-- Переменная для хранения хэндла рабочего окна.
-- Переменная глобальная!
-- Это не number, а void*
-- Для преобразования number в
-- корректный тип используйте:
-- window = tohandle(number)
_G.window = wnd.tohandle(0)


-- prev_handle, err, verbose = workwindow(path, name, class, pos, pid)
-- prev_handle, err, verbose = workwindow(handle)
-- Устанавливает рабочее окно
-- (глобальная переменная window),
-- если переданы параметры нового
-- рабочего окна.
-- Возвращает текущее рабочее окно.
--
-- prev_handle - предыдущий хэндл рабочего окна
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- вариант 1
-- handle - можеть быть задан хэндлом (cdata void*)
--          либо числом равным хэндлом (number),
--          name, class, pos, pid - игнорируются.
--
-- вариант 2
-- path  - путь к исполняемому файлу (exe).
--         Задается, как паттерн для regexp.
--         Проверка происходит в полном пути.
--
-- name  - заголовок окна. Задается как паттерн
--         для regexp.
--
-- class - имя класса окна. Задается как паттерн
--         для regexp.
--
-- pid   - pid процесса.
--
-- Паттерны стандартные для match в lua.
-- Используйте '^' для обозначения
-- начала строки и '$' для конца строки.
-- Например:
-- "name"  - найдет все окна, где присутствует "name"
-- "^name" - найдет все окна, начинающиеся с "name"
-- "name$" - найдет все окна, заканчивающиеся на "name"
-- "^name$" - найдет все окна, с точным именм "name"
-- Так же допустимо использование ?, .*, [] и т.д.
--
-- Внимание!
-- В случае 3 варианта синтаксиса рабочим окном
-- будет сделано первое найденное.
-- Самое близкое к поверхности рабочего стола
-- (последнее активное).
-- Любой из параметров может быть nil для
-- исключения фильтрации по этому параметру.
do
    wnd.workwindow = function(path, name, class, pos, pid)
        local window_prev = window
        if type(path) == "cdata" then
            window = path
            return window_prev, 0, "ERROR_SUCCESS"
        end

        if type(path) == "number" then
            local err, verbose = nil
            window = wnd.tohandle(path)
            return window_prev, 0, "ERROR_SUCCESS"
        end

        local result, err, verbose = wnd.findwindow(path, name, class, pos, pid)
        if result then
            window = result[1].handle
            return window_prev, err, verbose
        else
            print("Warning, can't apply a workwindow.", err, verbose)
            return window_prev, err, verbose
        end
        return window_prev, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = windowpos([pos], [handle])
-- Функция перемещает окно в заданный pos,
-- если pos не задан просто вернет текущее положение.
--
-- result  - таблица с итоговым положением окна:
--           result.x - начальная X координата
--           result.y - начальная Y координата
--           result.width  - ширина окна
--           result.height - высота окна
--           если произошла ошибка, то вместо
--           таблицы вернет nil.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- pos     - таблица с желаемым положением
--           и размером окна.
--           Структура аналогична result.
--           result.x - начальная X координата
--           result.y - начальная Y координата
--           result.width  - ширина окна
--           result.height - высота окна
--           любое из полей может отсутствовать (nil).
--           Если какое либо поле отсутствует,
--           оно будет оставлено без изменений.
--
-- handle  - хэндл окна с которым будет
--           работать функция.
--           Тип данных - HWND (он же void *).
--           Для преобразования числа в HWND
--           используйте функцию tohandle(number).
--           Если handle не задан - будет работать
--           с глобальной переменной window,
--           которая так же должна быть HWND.
--           window = tohandle(number)
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     HWND;
        BOOL MoveWindow(HWND hwnd, int X, int Y, int nWidth, int nHeight, BOOL bRepaint);
        ]]
    define[[
        typedef struct{
            int left;
            int top;
            int right;
            int bottom;
        } RECT;
    ]]
    ffi.cdef[[
        BOOL GetWindowRect(HWND hwnd, RECT *lpRect); // Получаем прямоугольник окна
        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    local rect = ffi.new("RECT")

    local move_window = function(handle, pos)
        return user32.MoveWindow(
            handle,
            pos.x, pos.y,
            pos.width, pos.height,
            0) -- no redraw
    end

    local get_window_rect = function(handle) -- Получаем прямоугольник окна
        local err = user32.GetWindowRect(handle, rect);  -- Получаем прямоугольник окна
        if err ~= 0 then
            return {
                    x = rect.left; -- Координата X верхнего левого угла окна
                    y = rect.top;  -- Координата Y верхнего левого угла окна
                    width = rect.right - rect.left;  -- Ширина окна
                    height = rect.bottom - rect.top; -- Высота окна
                }, 0, "ERROR_SUCCESS"
        else
            return nil, sys.geterr()
        end
    end

    wnd.windowpos = function(pos, handle)
        -- Работаем по хендлу или с рабочим окном?
        handle = ffi.cast("HWND", handle)

        -- Передана таблица с координатами
        -- для изменения параметров окна.
        if type(pos) == "table" then
            -- передана только часть паараметров?
            if not pos.x or not pos.y or not pos.width or not pos.height then
                local pos_current = get_window_rect(handle)
                pos.x      = pos.x      or pos_current.x
                pos.y      = pos.y      or pos_current.y
                pos.width  = pos.width  or pos_current.width
                pos.height = pos.height or pos_current.height
            end
            handle = handle or window
            if move_window(handle, pos) == 0 then
                return nil, geterr()
            end
            return pos, 0, "ERROR_SUCCESS"
        else
        -- Таблицы с параметрами окна нет.
        -- Возвращаем текущее положение окна.
            handle = pos or window
            local result, err, verbose = get_window_rect(handle)
            if err ~= 0 then
                return nil, err, verbose
            end
            return result, 0, "ERROR_SUCCESS"
        end
    end
end


-- new_handle, err = getwindow(direction, handle)
-- Получает хэндл окна относительно
-- handle в направлении direction.
--
-- new_handle - хэндл новго окна. Если произошла
--              ошибка - вернет nil.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- direction  - направление поиска окна.
--              допустимы следующие значения:
--              "first"
--              "last"
--              "next"
--              "prev"
--              "owner"
--              "child"
--              "popup"
do
    ffi.cdef[[
        typedef          void*     HWND;
        typedef unsigned int       UINT;
        HWND GetWindow(HWND hwnd, UINT uCmd);

        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    local dir = {}
    dir.first = 0 -- GW_HWNDFIRST
    dir.last  = 1 -- GW_HWNDLAST
    dir.next  = 2 -- GW_HWNDNEXT
    dir.prev  = 3 -- GW_HWNDPREV
    dir.owner = 4 -- GW_OWNER
    dir.child = 5 -- GW_CHILD
    dir.popup = 6 -- GW_ENABLEDPOPUP
    local NULL_POINTER    = ffi.cast("void*", 0)

    wnd.getwindow = function(direction, handle)
        handle = handle or window
        if not dir[direction] then
            return nil, 100001, "wrong direction"
        end
        local result = user32.GetWindow(handle, dir[direction])
        if result == NULL_POINTER then
            return nil, geterr()
        end
        return result, 0, "ERROR_SUCCESS"
    end
end


-- text, err, verbose = getwindowtext([handle])
-- Получает текст заголовка окна.
--
-- text    - текст заголовка окна.
--           Если ошибка - врент nil.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- handle  - хэндл окна. По умлчанию window.
do
    ffi.cdef[[
        typedef          void*     HWND;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;
        int GetWindowTextA(HWND hwnd, LPSTR lpString, int nMaxCount);
        int GetWindowTextLengthA(HWND hwnd);

        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    -- Создаем буфер
    local buffer_size = 1024
    local buffer = ffi.new("char[?]", buffer_size)
    wnd.getwindowtext = function(handle)
        handle = handle or window
        local lenght = user32.GetWindowTextLengthA(handle)
        -- Размер получить не удалось
        if lenght == 0 then
            return nil, geterr()
        end
        -- +1 на ноль в конце строки
        lenght = lenght + 1
        -- Увеличиваем буфер, если необходимо
        if lenght > buffer_size then
            buffer_size = 2^math.ceil(math.log(lenght, 2))
            buffer = ffi.new("char[?]", buffer_size)
        end

        local copied = user32.GetWindowTextA(handle, buffer, buffer_size)
        if copied == 0 then
            return nil, geterr()
        end
        return ffi.string(buffer), 0, "ERROR_SUCCESS"
    end
end


-- text, err, verbose = getwindowclass([handle])
-- Получает имя класса окна.
--
-- text   - имя класса окна.
--          Если ошибка - врент nil.
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- handle - хэндл окна. По умлчанию window.
do
    ffi.cdef[[
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;
        typedef          LPSTR     LPTSTR;
        int GetClassNameA(HWND hWnd, LPTSTR lpClassName, int nMaxCount);

        typedef unsigned int DWORD;
        DWORD GetLastError();
    ]]
    -- Создаем буфер
    local buffer_size = 1024
    local buffer = ffi.new("char[?]", buffer_size)
    wnd.getwindowclass = function(handle)
        handle = handle or window
        repeat
            local lenght = user32.GetClassNameA(handle, buffer, buffer_size)
            -- Размер получить не удалось
            if lenght == 0 then
                return nil, geterr()
            end
            -- +1 на ноль в конце строки
            lenght = lenght + 1
            -- Увеличиваем буфер, если необходимо
            if lenght > buffer_size then
                buffer_size = buffer_size*2
                buffer = ffi.new("char[?]", buffer_size)
            end
        until lenght < buffer_size

        return ffi.string(buffer), 0, "ERROR_SUCCESS"
    end
end


-- text = getcontroltext(handle)
-- Получает текст контрола:
-- поля ввода, имена кнопок и т.д.
--
-- text   - текст контрола
--
-- handle - хэндл контрола текст
--          которого необходимо получить
do
    local WM_GETTEXT       = 0x000D
    local WM_GETTEXTLENGTH = 0x000E

    -- Создаем буфер
    local buffer_size = 1024
    local buffer = ffi.new("char[?]", buffer_size)

    wnd.getcontroltext = function(handle)
        handle = handle or window
        local lenght = wnd.sendmessage(WM_GETTEXTLENGTH, 0, 0, handle)
        -- Размер получить не удалось
        if lenght == 0 then
            return nil
        end
        -- +1 на ноль в конце строки
        lenght = lenght + 1
        -- Увеличиваем буфер, если необходимо
        if lenght > buffer_size then
            buffer_size = 2^math.ceil(math.log(lenght, 2))
            buffer = ffi.new("char[?]", buffer_size)
        end
        lenght = wnd.sendmessage(WM_GETTEXT, buffer_size, buffer, handle)
        return ffi.string(buffer)
    end
end


-- err, verbose = setwindowtext([text, [handle]])
-- Установит новый заголовок для окна handle.
--
-- err     - вернет nil, если ошибка, true - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- text    - текст новго заголовка. Может быть nil.
--
-- handle  - хэндл окна имя которого будет изменяться.
--           По умолчанию window.
do
   ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     HWND;
        typedef const    CHAR*     LPCSTR;
        BOOL SetWindowTextA(HWND   hwnd, LPCSTR lpString);
        DWORD GetLastError();
    ]]

    wnd.setwindowtext = function(text, handle)
        handle = handle or window
        local result = user32.SetWindowTextA(handle, text)
        if result == 0 then
            return nil, geterr()
        end
        return true, 0, "ERROR_SUCCESS"
    end
end


-- lenght = setcontroltext(text, handle)
-- Получает текст контрола:
-- поля ввода, имена кнопок и т.д.
--
-- lenght - длина установленного заголовка.
--
-- text   - текст контрола.
--
-- handle - хэндл контрола текст
--          которого необходимо получить.
do
    local WM_SETTEXT = 0x000C
    wnd.setcontroltext = function(text, handle)
        handle = handle or window
        local lenght = wnd.sendmessage(WM_SETTEXT, 0, text, handle)
        return lenght
    end
end


-- result = isvisible(handle)
-- Проверяет является ли окно видимым.
-- Проверяется именно флаг невидимости.
-- Данная функция не предназачена для проверки
-- окна на перекрытие другими окнами.
--
-- result - Результат проверки. true - видимое, false - скрыто.
--
-- handle - хэндл проверяемого окна.
do
    ffi.cdef[[
        typedef          int       BOOL;
        BOOL IsWindowVisible(HWND hwnd);
    ]]
    wnd.isvisible = function(handle)
        return user32.IsWindowVisible(handle) ~= 0 and true or false
    end
end


-- result, err, verbose = findwindow(path, [name, [class, [pos, [pid]]]])
-- Функция ищет окна по заданным параметрам.
-- Каждый из параметров применяется, как фильтр.
-- Любой из параметров можно задать равным
-- nil для того, чтобы исключить из фильтрации.
-- Например:
-- result = findwindow(nil, name, class, nil, nil)
-- будет осуществлять поиск только по имени окна и класса.
--
-- result             - массив в котором содержатся результаты.
-- result[1]          - первое найденное окно.
-- result[2]          - второе найденное окно.
-- result[...]        - ...... найденное окно.
-- result[N]          -   N    найденное окно.
-- result[N].handle   - хэндл окна
-- result[N].name     - имя окна
-- result[N].class    - имя класса окна
-- result[N].x        - x координата левой границы окна
-- result[N].y        - y координата верхней границы окна
-- result[N].width    - ширина окна
-- result[N].height   - высота окна
-- result[N].pid      - pid окна
-- result[N].tid      - tid окна
-- result[N].exe      - имя файла
-- result[N].path     - путь к каталогу с файлом
-- result[N].fullpath - полный путь к exe
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- path    - путь к исполняемому файлу (exe).
--           Задается, как паттерн для regexp.
--           Проверка происходит в полном пути.
--
-- name    - заголовок окна. Задается как паттерн
--           для regexp.
--
-- class   - имя класса окна. Задается как паттерн
--           для regexp.
--
-- pos            - таблица с положением и размером окна.
-- pos.min_x      - начальная X координата области поиска
-- pos.min_y      - начальная Y координата области поиска
-- pos.min_width  - минимальная ширина окна
-- pos.min_height - минимальная высота окна
-- pos.max_x      - конечная  X координата области поиска
-- pos.max_y      - конечная  Y координата области поиска
-- pos.max_width  - максимальная ширина окна
-- pos.max_height - максимальная высота окна
--
-- pid     - pid процесса.
--
-- Паттерны стандартные для match в lua.
-- Используйте '^' для обозначения
-- начала строки и '$' для конца строки.
-- Например:
-- "name"  - найдет все окна, где присутствует "name"
-- "^name" - найдет все окна, начинающиеся с "name"
-- "name$" - найдет все окна, заканчивающиеся на "name"
-- "^name$" - найдет все окна, с точным именм "name"
-- Так же допустимо использование ?, .*, [] и т.д.
do
    if ffi.arch == "x64" then
        ffi.cdef[[
            typedef      __int64   LONG_PTR;
        ]]
    else
        ffi.cdef[[
            typedef      long      LONG_PTR;
        ]]
    end
    ffi.cdef[[
        typedef          void*     HWND;
        typedef          int       BOOL;
        typedef          LONG_PTR  LPARAM;
        typedef int (__stdcall *WNDENUMPROC)(HWND, LPARAM);
        BOOL EnumWindows(WNDENUMPROC lpEnumFunc, LPARAM lParam);
    ]]
    local window = nil

    local function enum_windows_callback(hwnd, _)
        window[#window+1] = hwnd
        return 1
    end

    wnd.findwindow = function(path, name, class, pos, pid)
        path  = sys.ispilot and name or sys.utf8_to_ascii(path)
        name  = sys.ispilot and name or sys.utf8_to_ascii(name)
        class = sys.ispilot and name or sys.utf8_to_ascii(class)

        local min_x      = pos and (pos.min_x      or pos.x      or nil)
        local min_y      = pos and (pos.min_y      or pos.y      or nil)
        local min_height = pos and (pos.min_height or pos.height or nil)
        local min_width  = pos and (pos.min_width  or pos.width  or nil)
        local max_x      = pos and (pos.max_x      or pos.x      or nil)
        local max_y      = pos and (pos.max_y      or pos.y      or nil)
        local max_height = pos and (pos.max_height or pos.height or nil)
        local max_width  = pos and (pos.max_width  or pos.width  or nil)

        -- Собираем все хэндлы.
        -- Решение перебирать окна здесь, по всем хэндлам,
        -- а не внутри callback функции обосновано тем, что
        -- мы все равно из callback не сможем вернуть lua массив.
        -- При этом в callback еще пришлось бы сишными
        -- структурами передавать name, class, pos, pid.
        -- Стык сред. Беда.
        -- window объявлен выше, это upvalue, а не global.
        window = {}
        local result = {}

        -- Результат enum в window
        ffi.C.EnumWindows(enum_windows_callback, 0)
        -- Фильтруем полученные окна.
        for i = 1, #window do
            -- Создаем переменные upvalue
            -- Чтобы при успешном поиске не дергать
            -- данные второй раз.
            local name_current  = nil
            local class_current = nil
            local pos_current   = nil
            local pid_current   = nil
            local tid           = nil
            local path_current  = nil
            local err           = nil
            local verbose       = nil

            -- Создаем флаг корректности окна.
            local fine = true
            -- Окно видимое?
            if fine and not wnd.isvisible(window[i]) then
                fine = false
            end
            -- Фильтр по заголовку включен?
            if fine and name then
                name_current, err, verbose = wnd.getwindowtext(window[i])
                if err ~= 0 then return nil, err, verbose end
                if not name_current or not name_current:match(name) then
                    fine = false
                end
            end
            -- Фильтр имени класса включен?
            if fine and class then
                class_current, err, verbose = wnd.getwindowclass(window[i])
                if err ~= 0 then return nil, err, verbose end
                if not class_current or not class_current:match(class) then
                    fine = false
                end
            end
            -- Размеры ограничены?
            if fine and pos then
                pos_current, err, verbose = wnd.windowpos(window[i])
                if err ~= 0 then return nil, err, verbose end
                if (min_x and min_x > pos_current.x + pos_current.width)
                or (max_x and max_x < pos_current.x)
                or (min_y and min_y > pos_current.y + pos_current.height)
                or (min_y and max_y < pos_current.y)
                or (min_width  and min_width  >= pos_current.width)
                or (max_width  and max_width  <= pos_current.width)
                or (min_height and min_height >= pos_current.height)
                or (max_height and max_height <= pos_current.height)
                then
                    fine = false
                end
            end
            -- pid задан?
            if fine and pid then
                pid_current, tid, err, verbose = wnd.windowpid(window[i])
                if err ~= 0 then return nil, err, verbose end
                if pid_current ~= pid then
                    fine = false
                end
            end
            -- Фильтр по исполняемому файлу включен?
            if fine and path then
                path_current, err, verbose = wnd.exepath(pid_current or wnd.windowpid(window[i]))
                if err ~= 0 then return nil, err, verbose end
                if not path_current or not path_current.fullpath:match(path) then
                    fine = false
                end
            end

            if fine then
                if not pos_current then
                    pos_current, err, verbose = wnd.windowpos(window[i])
                    if err ~= 0 then return nil, err, verbose end
                end
                if not tid then
                    pid_current, tid, err, verbose = wnd.windowpid(window[i])
                    if err ~= 0 then return nil, err, verbose end
                end
                if not path_current then
                    path_current, err, verbose = wnd.exepath(
                        pid_current or wnd.windowpid(window[i]))
                    if err ~= 0 then return nil, err, verbose end
                end
                result[#result+1] = {
                        handle   = window[i],
                        name     = name_current  or wnd.getwindowtext (window[i]),
                        class    = class_current or wnd.getwindowclass(window[i]),
                        x        = pos_current.x,
                        y        = pos_current.y,
                        width    = pos_current.width,
                        height   = pos_current.height,
                        pid      = pid_current,
                        tid      = tid,
                        exe      = path_current.exe,
                        path     = path_current.path,
                        fullpath = path_current.fullpath,
                    }
            end
        end
        if #result == 0 then
            return nil,    0, "ERROR_SUCCESS"
        else
            return result, 0, "ERROR_SUCCESS"
        end
    end
end


-- Получаем информацию об окне по его хэндлу.
-- info, err, verbose = windowinfo(handle)
--
-- info - массив с информацией об окне.
--
-- info.handle   - хэндл окна
-- info.name     - имя окна
-- info.class    - имя класса окна
-- info.x        - x координата левой границы окна
-- info.y        - y координата верхней границы окна
-- info.width    - ширина окна
-- info.height   - высота окна
-- info.pid      - pid окна
-- info.tid      - tid окна
-- info.exe      - имя файла
-- info.path     - путь к каталогу с файлом
-- info.fullpath - полный путь к exe
--
-- err  - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
do
    wnd.windowinfo = function(handle)
        local name_current  = nil
        local class_current = nil
        local pos_current   = nil
        local pid_current   = nil
        local tid           = nil
        local path          = nil
        local err           = nil
        local verbose       = nil

        -- Получаем имя
        name_current,      err, verbose = wnd.getwindowtext (handle)
        if err ~= 0 then return nil, err, verbose end
        -- Получаем класс
        class_current,     err, verbose = wnd.getwindowclass(handle)
        if err ~= 0 then return nil, err, verbose end
        -- Получаем размер
        pos_current,       err, verbose = wnd.windowpos     (handle)
        if err ~= 0 then return nil, err, verbose end
        -- Полчучем pid, tid
        pid_current, tid,  err, verbose = wnd.windowpid   (handle)
        if err ~= 0 then return nil, err, verbose end
        -- Получаем путь к файлу
        path,              err, verbose = wnd.exepath       (handle)
        if err ~= 0 then return nil, err, verbose end

        local result = {
                handle   = handle,
                name     = name_current,
                class    = class_current,
                x        = pos_current.x,
                y        = pos_current.y,
                width    = pos_current.width,
                height   = pos_current.height,
                pid      = pid_current,
                tid      = tid,
                exe      = path.exe,
                path     = path.path,
                fullpath = path.fullpath,
            }
        return result, 0, "ERROR_SUCCESS"
    end
end


-- result = showwindow([state, [handle]])
-- Изменяет состояние окна handle на состояние state
--
-- result - Если окно ранее было видимым, возвращаемое значение равно true.
--          Если окно ранее было скрыто , возвращаемое значение равно false.
--
-- handle - хэндл окна с которым производится действие. По умолчанию window.
-- state - состояние окна. По умолчанию na.
-- Допустимые значения кратко:
-- "na"            - вытащить на передний план.
-- "top"           - псевдоним для na.
-- "hide"          - скрыть
-- "maximize"      - развернуть на весь экран
-- "show"          - сделать видимым (анти-hide)
-- "minimize"      - свернуть
-- "restore"       - отменить сворачивание, отменить на весь экран
--
-- Полный набор флагов state и полная расшифровка:
-- "hide"          - Скрывает окно и активирует другое окно.
-- "normal"        - Активирует и отображает окно. Если окно свернуто,
--                   развернуто или упорядочено, система восстанавливает
--                   его исходный размер и положение. Приложение должно
--                   указать этот флаг при первом отображении окна.
-- "minimized"     - Активирует окно и отображает его как свернутое окно.
-- "maximize"      - Активирует окно и отображает его в виде развернутого окна.
-- "noactivate"    - Отображает окно с последним размером и положением.
--                   Это значение похоже на SW_SHOWNORMAL, за исключением
--                   того, что окно не активировано.
-- "show"          - Активирует окно и отображает его
--                   в текущем размере и положении.
-- "minimize"      - Свертывание указанного окна и активация следующего
--                   окна верхнего уровня в порядке Z.
-- "minnoactive"   - Отображает окно в виде свернутого окна.
--                   Это значение похоже на SW_SHOWMINIMIZED,
--                   за исключением того, что окно не активировано.
-- "na"            - Отображает окно в его текущем размере и положении.
--                   Это значение похоже на SW_SHOW, за исключением того,
--                   что окно не активировано.
-- "top"           - псевдоним для na.
-- "restore"       - Активирует и отображает окно. Если окно свернуто,
--                   развернуто или упорядочено, система восстанавливает
--                   его исходный размер и положение. Приложение должно
--                   указать этот флаг при восстановлении свернутого окна.
-- "default"       - Задает состояние отображения на основе значения SW_ ,
--                   указанного в структуре STARTUPINFO , переданной
--                   в функцию CreateProcess программой,
--                   которая запустила приложение.
-- "forceminimize" - Свертывание окна, даже если поток, которому
--                   принадлежит окно, не отвечает. Этот флаг следует
--                   использовать только при минимизации окон
--                   из другого потока.
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     HWND;
        BOOL ShowWindow(HWND hwnd, int  nCmdShow);
    ]]
    local SW_HIDE            = 0
    --local SW_SHOWNORMAL    = 1
    local SW_NORMAL          = 1
    local SW_SHOWMINIMIZED   = 2
    --local SW_SHOWMAXIMIZED = 3
    local SW_MAXIMIZE        = 3
    local SW_SHOWNOACTIVATE  = 4
    local SW_SHOW            = 5
    local SW_MINIMIZE        = 6
    local SW_SHOWMINNOACTIVE = 7
    local SW_SHOWNA          = 8
    local SW_RESTORE         = 9
    local SW_SHOWDEFAULT     = 10
    local SW_FORCEMINIMIZE   = 11
    wnd.showwindow = function(state, handle)
        state  = state  or SW_SHOWNA
        handle = handle or window
        if     state == "hide"          then
            state = SW_HIDE
        elseif state == "normal"        then
            state = SW_NORMAL
        elseif state == "minimized"     then
            state = SW_SHOWMINIMIZED
        elseif state == "maximize"      then
            state = SW_MAXIMIZE
        elseif state == "noactivate"    then
            state = SW_SHOWNOACTIVATE
        elseif state == "show"          then
            state = SW_SHOW
        elseif state == "minimize"      then
            state = SW_MINIMIZE
        elseif state == "minnoactive"   then
            state = SW_SHOWMINNOACTIVE
        elseif state == "na"            then
            state = SW_SHOWNA
        elseif state == "top"           then
            state = SW_SHOWNA
        elseif state == "restore"       then
            state = SW_RESTORE
        elseif state == "default"       then
            state = SW_SHOWDEFAULT
        elseif state == "forceminimize" then
            state = SW_FORCEMINIMIZE
        else
            return nil
        end
        return user32.ShowWindow(handle, state) ~= 0 and true or false
    end
end


-- handle, err, verbose = windowfromcursor(x, y)
-- Возвращает окно под координатами x, y.
--
-- handle  - хэндл найденного окна.
--           Если ошибка - вернет nil.
--
-- err     - код ошибки на этапе получения
--           координат мышки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- x       - x координата поиска.
--
-- y       - y координата поиска.
do
    ffi.cdef[[
        typedef          void*     HWND;
        typedef          long      LONG;
    ]]
    define[[
        typedef struct tagPOINT {
            LONG x;
            LONG y;
        } POINT, *PPOINT, *NPPOINT, *LPPOINT;
    ]]
    ffi.cdef[[
        HWND WindowFromPoint(POINT Point);

        typedef          int       BOOL;
        BOOL GetCursorPos(POINT *lpPoint);
        DWORD GetLastError();
    ]]
    local point = ffi.new("POINT")
    local NULL_POINTER    = ffi.cast("void*", 0)

    wnd.windowfromcursor = function()
        local err = user32.GetCursorPos(point)
        -- Нет координат мышки.
        if err == 0 then
            return nil, geterr()
        end
        local handle = user32.WindowFromPoint(point)
        if handle == NULL_POINTER then
            return nil, 0, "ERROR_SUCCESS"
        end
        return handle, 0, "ERROR_SUCCESS"
    end
end


-- handle = windowfrompoint(x, y)
-- Возвращает окно под координатами x, y.
--
-- handle - хэндл найденного окна.
--          Если ошибка - вернет nil.
--
-- x      - x координата поиска.
--
-- y      - y координата поиска.
do
    ffi.cdef[[
        typedef          void*     HWND;
        typedef          long      LONG;
    ]]
    define[[
        typedef struct tagPOINT {
            LONG x;
            LONG y;
        } POINT, *PPOINT, *NPPOINT, *LPPOINT;
    ]]
    ffi.cdef[[
        HWND WindowFromPoint(POINT Point);
        HWND ChildWindowFromPoint(HWND hWndParent, POINT Point);
    ]]
    local point = ffi.new("POINT")
    local NULL_POINTER    = ffi.cast("void*", 0)

    wnd.windowfrompoint = function(x, y)
        point.x = x
        point.y = y
        local handle = user32.WindowFromPoint(point)
        if handle == NULL_POINTER then
            return nil
        end
        return handle
    end
end


-- pid, tid, err, verbose = windowpid([handle])
-- Вернет pid и tid процесса и потока.
--
-- pid     - pid процесса которому принадлежит окно.
--
-- tid     - tid потока который создал окно.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- handle  - хэндл окна данные которого запрашиваются.
--           По умолчанию window.
do
    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          void*     HWND;
        typedef          DWORD*    LPDWORD;
        DWORD GetWindowThreadProcessId(HWND hwnd, LPDWORD lpdwProcessId);
        DWORD GetLastError();
    ]]
    local PID = ffi.new("DWORD[1]")

    wnd.windowpid = function(handle)
        handle = handle or window
        local tid = user32.GetWindowThreadProcessId(handle, PID)
        if tid == 0 then
            return nil, nil, geterr()
        end
        return PID[0], tid, 0, "ERROR_SUCCESS"
    end
end


-- result = sendmessage([message, [wparam, [lparam, [handle]]]])
-- Отсылка сообщений в окна.
--
-- result  - код результата. Индивидуален для message.
--
-- message - сообщение. По умолчанию 0.
--
-- wparam  - wparam параметр сообщения. По умолчанию 0.
--
-- lparam  - lparam параметр сообщения. По умолчанию 0.
--
-- handle  - хэндл окна. Если не указать - работает с window.
--
-- Все параметры опциональные.
-- За что именно отвечают wparam и lparam индивидуально
-- для каждого сообщения.
do
    if ffi.arch == "x64" then
        ffi.cdef[[
            typedef unsigned __int64  UINT_PTR;
            typedef          __int64  LONG_PTR;
        ]]
    else
        ffi.cdef[[
            typedef unsigned int   UINT_PTR;
            typedef          long  LONG_PTR;
        ]]
    end
    ffi.cdef[[
        typedef          LONG_PTR  LRESULT;
        typedef          void*     HWND;
        typedef unsigned int       UINT;
        typedef          UINT_PTR  WPARAM;
        typedef          LONG_PTR  LPARAM;

        LRESULT SendMessageA(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam);
    ]]
    wnd.sendmessage = function(message, wparam, lparam, handle)
        handle  = handle  or window
        message = message or 0
        wparam  = wparam  or 0
        lparam  = lparam  or 0
        local result = user32.SendMessageA(handle, message, wparam, lparam)
        return result
    end
end


-- result = postmessage([message, [wparam, [lparam, [handle]]]])
-- Отсылка сообщений в окна.
--
-- result  - код результата. Индивидуален для message.
--
-- message - сообщение. По умолчанию 0.
--
-- wparam  - wparam параметр сообщения. По умолчанию 0.
--
-- lparam  - lparam параметр сообщения. По умолчанию 0.
--
-- handle  - хэндл окна. Если не указать - работает с window.
--
-- Все параметры опциональные.
-- За что именно отвечают wparam и lparam индивидуально
-- для каждого сообщения.
do
    if ffi.arch == "x64" then
        ffi.cdef[[
            typedef unsigned __int64  UINT_PTR;
            typedef          __int64  LONG_PTR;
        ]]
    else
        ffi.cdef[[
            typedef unsigned int   UINT_PTR;
            typedef          long  LONG_PTR;
        ]]
    end
    ffi.cdef[[
        typedef          LONG_PTR  LRESULT;
        typedef          void*     HWND;
        typedef unsigned int       UINT;
        typedef          UINT_PTR  WPARAM;
        typedef          LONG_PTR  LPARAM;

        LRESULT PostMessageA(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam);
    ]]
    wnd.postmessage = function(message, wparam, lparam, handle)
        handle  = handle  or window
        message = message or 0
        wparam  = wparam  or 0
        lparam  = lparam  or 0
        local result = user32.PostMessageA(handle, message, wparam, lparam)
        return result
    end
end


-- handle = GetForegroundWindow()
-- Получает хэндл активного окна.
--
-- handle - хэндл активного окна.
do
    ffi.cdef[[
        HWND GetForegroundWindow();
    ]]
    wnd.getforeground = function()
        return user32.GetForegroundWindow()
    end
end


-- handle, err, verbose = getfocus()
-- Получает хэндл окна в который
-- осуществляется ввод с клавитуры.
-- (в котором мигает каретка).
--
-- handle  - хэндл окна с фокусуом клавиватуры.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки.
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     HWND;
        typedef unsigned int       DWORD;

        HWND GetForegroundWindow();
        DWORD GetCurrentThreadId();

        DWORD GetLastError();
        BOOL AttachThreadInput(
            DWORD idAttach,
            DWORD idAttachTo,
            BOOL  fAttach
        );
    ]]
    define[[
        typedef struct{
            int left;
            int top;
            int right;
            int bottom;
        } RECT;
        typedef struct tagGUITHREADINFO {
            DWORD cbSize;
            DWORD flags;
            HWND  hwndActive;
            HWND  hwndFocus;
            HWND  hwndCapture;
            HWND  hwndMenuOwner;
            HWND  hwndMoveSize;
            HWND  hwndCaret;
            RECT  rcCaret;
        } GUITHREADINFO, *PGUITHREADINFO, *LPGUITHREADINFO;
    ]]
    ffi.cdef[[
        BOOL GetGUIThreadInfo(
            DWORD          idThread,
            PGUITHREADINFO pgui
        );

        HWND GetFocus();
    ]]
    local GUITHREADINFO  = ffi.new  ("GUITHREADINFO")
    GUITHREADINFO.cbSize = ffi.sizeof(GUITHREADINFO)
    wnd.getfocus = function()
        local foreground           = user32.GetForegroundWindow()
        local _, tid, err, verbose = wnd.windowpid(foreground)
        if err                    ~= 0 then return nil, err, verbose end
        local current_tid          = kernel32.GetCurrentThreadId();
        local try_to_attach        = user32.AttachThreadInput(current_tid, tid, true);
        if try_to_attach          == 0 then return nil, geterr() end
        local try_to_get_info      = user32.GetGUIThreadInfo(tid, GUITHREADINFO)
        if try_to_get_info        == 0 then return nil, geterr() end
        local cancel_attach        = user32.AttachThreadInput(current_tid, tid, false);
        if cancel_attach          == 0 then return nil, geterr() end

        return GUITHREADINFO.hwndFocus, 0, "ERROR_SUCCESS"
    end
end


-- Таблица для с функциями обеспечивающими
-- совместимость со старым синтаксисом.
--
-- При необходимости просто переобъявите.
-- Например, windowpos = wnd.compatibility.windowpos
--
-- wnd.compatibility.workwindow
-- не модифицирует каждую функцию переводя ее на
-- использование workwindow.
-- Он начинает применять переданный
-- хэндл к глобальной window.
wnd.compatibility = {}
do
    local workwindow_back = workwindow
    wnd.compatibility.workwindow = function(handle)
        local result = workwindow()
        if handle then
            local old_style_handle = workwindow_back(handle)
            window = wnd.tohandle(old_style_handle)
        end
        return result
    end
end


wnd.compatibility.windowpos = function(x, y, width, height, handle)
    local pos  = {x=x, y=y, width=width, height=height}
    pos.width  = pos.width  ~= -1 and pos.width  or nil
    pos.height = pos.height ~= -1 and pos.height or nil
    handle = handle and wnd.tohandle(handle) or window or workwindow()
    return wnd.windowpos(pos, handle)
end

do
    wnd.compatibility.getwindow = function(handle, direction)
        handle = ffi.cast("HWND", handle)
        return wnd.getwindow(direction, handle)
    end
end

do
    wnd.compatibility.setwindowtext = function(handle, text)
        if not handle or not text then
            return 0
        end
        local result = wnd.setwindowtext(text, handle)
        return result == true and 1 or 0
    end
end

do
    wnd.compatibility.findwindow = function(target)
        local result = nil
        if target then
            result = wnd.findwindow(nil, target)
            if not result then
                result = wnd.findwindow(nil, nil, nil, nil, target)
            end
        else
            result = wnd.getforeground()
        end
        return result
    end
end

do
    wnd.compatibility.showwindow = function(handle, state)
        wnd.showwindow(state, handle)
    end
end

do
    wnd.compatibility.workwindowpid = function()
        return wnd.windowpid(window)
    end
end

do
    wnd.compatibility.sendmessage = function(handle, message, wparam, lparam)
        return wnd.sendmessage(message, wparam, lparam, handle)
    end
end

do
    wnd.compatibility.postmessage = function(handle, message, wparam, lparam)
        return wnd.postmessage(message, wparam, lparam, handle)
    end
end











--     XXXXXXXXXXXX       XXXXXXXXXXXX          XXXXXXXX           XXXXXXXX
--     XXXXXXXXXXXXXX     XXXXXXXXXXXXXX      XXXXXXXXXXXX       XXXXXXXXXXXX
--     XXXX      XXXX     XXXX      XXXX     XXXXX    XXXXX     XXXXX    XXXXX
--     XXXX      XXXX     XXXX      XXXX     XXXX      XXXX     XXXX
--     XXXXXXXXXXXXXX     XXXXXXXXXXXXXX     XXXX      XXXX     XXXX
--     XXXXXXXXXXXX       XXXXXXXXXXXX       XXXX      XXXX     XXXX
--     XXXX               XXXX  XXXX         XXXX      XXXX     XXXX
--     XXXX               XXXX   XXXX        XXXX      XXXX     XXXX
--     XXXX               XXXX    XXXX       XXXXX    XXXXX     XXXXX    XXXXX
--     XXXX               XXXX     XXXX       XXXXXXXXXXXX       XXXXXXXXXXXX
--     XXXX               XXXX      XXXX        XXXXXXXX           XXXXXXXX










-- exec(str)
-- Запуск процесса без ожидания завершения
--
-- str - путь для запуска исполняемого файла.
do
    wnd.exec = function(str)
        return os.execute("start "..str)
    end
end


-- execandwait(str)
-- Запуск процесса с ожиданием его завершения.
--
-- str - путь для запуска исполняемого файла.
do
    wnd.execandwait = function(str)
        return os.execute(str)
    end
end


-- path, err, verbose = exepath([handle|pid])
-- Получает путь к исполняемому файлу.
--
-- path          - массив с результатом
-- path.exe      - имя файла
-- path.path     - путь к каталогу с файлом
-- path.fullpath - полный путь к exe
-- Пример:
-- path.exe      - cmd.exe
-- path.path     - c:\windows\system32\
-- path.fullpath - c:\windows\system32\cmd.exe
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- handle - handle процесса информацию о котором получаем
--          Значение по умолчанию window.
--
-- pid    - pid процесса информацию о котором получаем
--
-- Внимание! Указывается либо pid либо handle.
-- Тип данный pid - number, handle - HWND (void*).
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          DWORD*    PDWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;

        BOOL QueryFullProcessImageNameA(
            HANDLE hProcess,
            DWORD  dwFlags,
            LPSTR  lpExeName,
            PDWORD lpdwSize
        );

        HANDLE OpenProcess(
            DWORD dwDesiredAccess,
            BOOL  bInheritHandle,
            DWORD dwProcessId);

        BOOL CloseHandle(HANDLE hObject);
        DWORD GetLastError();
    ]]
    local PROCESS_QUERY_INFORMATION = 0x0400

    -- В связи с тем, что QueryFullProcessImageNameA
    -- перезаписывает переданный ему размер буфера,
    -- То приходится хранить переменную с реальным
    -- размером буфера и каждый раз записывать
    -- в массив значение переменной под убой.
    -- Тут видимо был новый рецепт наркоты
    -- и конкретный передоз. Браво майкрософт!
    -- При том, что QueryFullProcessImageNameA
    -- возвращает BOOL, который по факту так же
    -- занимает тот же размер что и значение
    -- размера буфера в современных реалиях,
    -- очень много вопросов вызвает в чем была проблема
    -- вместо BOOL возвращать количество копируемых
    -- символов, так же как и других функциях.

    local buffer_size     = 32768
    local buffer          = ffi.new("char[?]", buffer_size)
    local buffer_size_ptr = ffi.new("DWORD[1]")

    wnd.exepath = function(pid)
        pid = type(pid) == "number" and pid or wnd.windowpid(pid)

        local process_handle = kernel32.OpenProcess(
            PROCESS_QUERY_INFORMATION, 0, pid)
        if process_handle == 0 then return nil, geterr() end

        repeat
            buffer_size_ptr[0] = buffer_size
            local name_result = kernel32.QueryFullProcessImageNameA(
            process_handle, 0, buffer, buffer_size_ptr)
            -- Завершилось неудачей?
            -- 122 ошибка - нехватка буфера,
            -- при ее возникновении увеличиваем
            -- размер буфера, нет смысла прерывть функцию.
            local err = kernel32.GetLastError()
            if name_result == 0 and err ~= 122 then
                return nil, err, system_error[err]
            end

            -- +1 на ноль в конце строки
            local lenght = buffer_size_ptr[0] + 1
            -- Увеличиваем буфер, если необходимо
            if lenght > buffer_size then
                buffer_size = buffer_size*2
                buffer = ffi.new("char[?]", buffer_size)
            end
        until name_result ~= 0


        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, geterr() end

        local result    = {}
        result.fullpath = ffi.string(buffer)
        result.path, result.exe = result.fullpath:match("(.*\\)([^\\]*)")
        return result, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = terminate(victim, name, class, pos, pid)
-- Форсированно завершает процесс.
--
-- result  - успех выполнения true/nil.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- вариант 1
-- victim  - можеть быть задан хэндлом тип HWND (cdata void*)
--           name, class, pos, pid - игнорируются.
--
-- вариант 2
--           Любой из параметров может быть nil,
--           для исключения его из фильтрации.
--
-- victim  - путь к исполняемому файлу (exe).
--           Задается, как паттерн для regexp.
--           Проверка происходит в полном пути.
--
-- name    - заголовок окна. Задается как паттерн
--           для regexp.
--
-- class   - имя класса окна. Задается как паттерн
--           для regexp.
--
-- pos            - таблица с положением и размером окна.
-- pos.min_x      - начальная X координата области поиска
-- pos.min_y      - начальная Y координата области поиска
-- pos.min_width  - минимальная ширина окна
-- pos.min_height - минимальная высота окна
-- pos.max_x      - конечная  X координата области поиска
-- pos.max_y      - конечная  Y координата области поиска
-- pos.max_width  - максимальная ширина окна
-- pos.max_height - максимальная высота окна
--
-- pid     - pid процесса.
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef unsigned int       UINT;
        typedef          void*     HWND;
        typedef          DWORD*    LPDWORD;
        DWORD GetWindowThreadProcessId(HWND hwnd, LPDWORD lpdwProcessId);
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        DWORD GetLastError();
        BOOL TerminateProcess(HANDLE hProcess, UINT uExitCode);
        BOOL CloseHandle(HANDLE hObject);
    ]]
    local PROCESS_TERMINATE = 0x0001
    wnd.terminate = function(victim, name, class, pos, pid)
        local pid_to_kill = {}
        if type(victim) == "cdata" then
            local _        = nil
            local err      = nil
            local verbose  = nil
            pid_to_kill[1] = {}
            pid_to_kill[1].pid, _, err, verbose = wnd.windowpid(victim)
            if err ~= 0 then return nil, err, verbose end
        elseif pid then
            pid_to_kill[1] = {}
            pid_to_kill[1].pid = pid
        else
            local result, err, verbose = wnd.findwindow(victim, name, class, pos, pid)
            if result then
                pid_to_kill = result
            else
                return nil, err, verbose
            end
        end

        for i = 1, #pid_to_kill do
            local process_handle   = kernel32.OpenProcess(
                PROCESS_TERMINATE, 0, pid_to_kill[i].pid)
            if process_handle     == 0 then return nil, geterr() end
            local terminate_result = kernel32.TerminateProcess(process_handle, 0)
            if terminate_result   == 0 then return nil, geterr() end
            local close_result     = kernel32.CloseHandle(process_handle)
            if close_result       == 0 then return nil, geterr() end
        end
        return true, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = setpriority(prio, [handle|pid])
-- Устанавливает приоритет для цли.
--
-- result - успех выполнения true/false.
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- prio   - желаемый приоритет. Возможные варианты:
--          "real"   - реального времени
--          "high"   - высокий
--          "above"  - выше среднего
--          "normal" - средний
--          "below"  - ниже среднего
--          "idle"   - низкий
--
-- handle - хэндл процесса HWND (void*)
--          По умолчанию window.
--
-- pid    - pid процесса (number).
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        BOOL SetPriorityClass(
            HANDLE hProcess,
            DWORD  dwPriorityClass
        );
        DWORD GetLastError();
    ]]
    local PROCESS_SET_INFORMATION = 0x0200
    local prio_list = {}
    prio_list.real   = 0x00000100 -- REALTIME_PRIORITY_CLASS
    prio_list.high   = 0x00000080 -- HIGH_PRIORITY_CLASS
    prio_list.above  = 0x00008000 -- ABOVE_NORMAL_PRIORITY_CLASS
    prio_list.normal = 0x00000020 -- NORMAL_PRIORITY_CLASS
    prio_list.below  = 0x00004000 -- BELOW_NORMAL_PRIORITY_CLASS
    prio_list.idle   = 0x00000040 -- IDLE_PRIORITY_CLASS

    wnd.setpriority = function(prio, target)
        target = target or window
        if type(target) == "cdata" then
            target = wnd.windowpid(target)
        end

        local process_handle = kernel32.OpenProcess(PROCESS_SET_INFORMATION, 0, target)
        if process_handle   == 0 then return nil, geterr() end
        local prio_result    = kernel32.SetPriorityClass(process_handle, prio_list[prio])
        if prio_result      == 0 then return nil, geterr() end
        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, geterr() end
        return true, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = setaffinity(mask, [handle|pid])
-- Устанавливает приоритет для цли.
--
-- result - успех выполнения true/false.
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- mask   - битовая маска соответствия с процессорами.
--          Может быть задана числом в виде
--          number или ULONG_PTR.
--          Внимание number может корректно хранить
--          только 52 бита, т.е. 52 процессора.
--          Альтернативный способ задать маску
--          строка с символами "Y", где
--          позиция = номер процессора.
--          Прочие символы игнорируются.
--          Пример:
--          "01YY456Y" - задаст соответствие
--          2, 3, 7 процессорам (отсчет с нуля).
--          Цифры используются только для удобства
--          в качестве разделителя.
--
-- handle - хэндл процесса HWND (void*)
--          По умолчанию window.
--
-- pid    - pid процесса (number).
do
    if ffi.arch == "x64" then
        ffi.cdef[[
                typedef unsigned __int64 ULONG_PTR;
            ]]
    else
        ffi.cdef[[
                typedef unsigned long ULONG_PTR;
            ]]
    end
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          ULONG_PTR DWORD_PTR;
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        BOOL SetProcessAffinityMask(
            HANDLE    hProcess,
            DWORD_PTR dwProcessAffinityMask
        );
        DWORD GetLastError();
    ]]
    local PROCESS_SET_INFORMATION = 0x0200
    local mask = ffi.new("ULONG_PTR[1]")

    -- mask объявлено выше т.к. завист от битности.
    local parse_mask = function(str)
        for i = 1, #str do
            if str:sub(i, i) == "Y" then
                mask = mask + 2^i
            end
        end
        return mask
    end

    wnd.setaffinity = function(new_mask, target)
        if type(new_mask) == "string" then
            mask[0] = parse_mask(new_mask)
        else
            mask[0] = new_mask
        end

        target = target or window
        if type(target) == "cdata" then
            target = wnd.windowpid(target)
        end

        local process_handle = kernel32.OpenProcess(PROCESS_SET_INFORMATION, 0, target)
        if process_handle   == 0 then return nil, geterr() end
        local aff_result     = kernel32.SetProcessAffinityMask(process_handle, mask[0])
        if aff_result       == 0 then return nil, geterr() end
        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, geterr() end
        return true, 0, "ERROR_SUCCESS"
    end
end

return wnd


