--lua
--[[
    Copyright (c) <2021>, <DarkMaster forum.uokit.com> All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    Usage is only allowed in not modifed UOPilot (uopilot.uokit.com) program,
    use in other programs is allowed only with the written permission of the owner.
    Redistribution of the original or modified code must be free, sale or commercial
    gain in any form is possible only with the written permission of the owner.
    Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
    Neither the name of the owner nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
    OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    OF THE POSSIBILITY OF SUCH DAMAGE.
]]

local ffi           = require[[ffi]]

local sys           = require[[lua_system\system]]
-- предотвращаем вечный цикл взаимной загрузки
package.loaded[ [[lua_system\proc]] ] = {}
local wnd           = require[[lua_system\window]]

local kernel32      = ffi.load("kernel32")
local user32        = ffi.load("user32"  )








local proc          = {}











-- dir = cd([path])
-- Изменяет рабочий каталог.
--
-- dir  - рабочий каталог после вызова cd
--
-- path - путь к новому рабочему каталогу.
--        Если не указан, то функция просто
--        вернет текущий рабочий каталог.
--        По умолчанию: nil.
do
    ffi.cdef[[
        typedef int                BOOL;
        typedef          char      CHAR;
        typedef const    CHAR*     LPCSTR; // must be "__nullterminated", "CONST"
        typedef LPCSTR             LPCTSTR;
        BOOL SetCurrentDirectoryA(
            LPCTSTR lpPathName
        );

        typedef unsigned int       DWORD;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;
        typedef          LPSTR     LPTSTR;
        DWORD GetCurrentDirectoryA(
            DWORD  nBufferLength,
            LPTSTR lpBuffer
        );
    ]]

    local buffer_size = 32768
    local buffer      = ffi.new("char[?]", buffer_size)

    proc.cd = function(path)
        -- Устанавливаем директорию.
        if path then
            local result = kernel32.SetCurrentDirectoryA(path)
            if result == 0 then
                return nil, sys.geterr()
            end
        end

        -- Запрашиваем размер строки.
        local size = kernel32.GetCurrentDirectoryA(0, nil);
        if size == 0 then
            return nil, sys.geterr()
        end
        -- Проверяем достаточность буфера.
        if buffer_size < size then
            buffer_size = 2^math.ceil(math.log(size, 2))
            buffer = ffi.new("char[?]", buffer_size)
        end

        -- Получаем директорию.
        local len = kernel32.GetCurrentDirectoryA(buffer_size, buffer);
        if len == 0 then
            return nil, sys.geterr()
        end

        return ffi.string(buffer, len), 0, "ERROR_SUCCESS"
    end
end

-- data, err, verbose = getenvironment(str_flag)
-- Получает все переменные окружения.
-- data     - результат. Если str_flag не задан или false,
--            то содержит lua таблицу где имя переменной
--            является ключем, а значение соответственно
--            значением. Т.е.
--            data[name] = value
--            Если включен флаг str_flag, то вернет строку
--            на строку в формате переменная=значение\0
--            и дополнительный \0 в конце строки.
--            Это исходный формат данных.
--
-- err      - код ошибки. 0 - успех.
--
-- verbose  - текстовая расшифровка ошибки.
--
-- str_flag - возвращать результат в виде строки.
--            По умолчанию false.
do
    -- ExpandEnvironmentStringsA
    ffi.cdef[[
        typedef unsigned char*     LPCH;
        LPCH GetEnvironmentStrings();
    ]]
    proc.getenvironment = function(str_flag)
        local c_str = kernel32.GetEnvironmentStrings()
        if c_str ~= 0 then
            -- Формат переменная=значение\0
            -- и дополнительный \0 в конце строки.
            -- Получаем размер.
            local size = 0
            for i = 0, 2^52 do
                if c_str[i] == 0 and c_str[i+1] == 0 then
                    size = i+1
                    break
                end
            end

            if str_flag then
                local result = ffi.string(c_str, size+1)
                return result, 0, "ERROR_SUCCESS"
            else
                -- Исключаем пустую строку из \x0\x0.
                if size > 1 then
                    local result = {}
                    local element = ""
                    local i = 0
                    while i < size-2 do
                        local element = ""
                        repeat
                            element = element..string.char(c_str[i])
                            i = i + 1
                        until c_str[i] == 0
                        i = i + 1
                        local key, value = element:match([[^(=?.*)=(.*)$]])
                        result[key] = value
                    end
                    return result, 0, "ERROR_SUCCESS"
                else
                    return nil, 0, "ERROR_SUCCESS"
                end
            end
        else
            return nil, sys.geterr()
        end
    end
end


-- exec(path)
-- Запуск процесса без ожидания завершения.
--
-- path - путь для запуска исполняемого файла.
do
    proc.exec = function(path)
        return os.execute("start "..path)
    end
end


-- execandwait(path)
-- Запуск процесса с ожиданием его завершения.
--
-- path - путь для запуска исполняемого файла.
do
    proc.execandwait = function(path)
        return os.execute(path)
    end
end


-- pipe_in, pipe_out, pipe_err, info = execpipe(path, [env], [combine])
-- Запускает процесс открывает pipe в обоих направлениях.
--
-- pipe_in             - объект для передачи данных приложению.
--                       pipe_in("some_data") - записать в пайп.
--                       pipe_in.handle       - хэндл пайпа
--
-- pipe_out            - объект для чтения данных передаваемых приложением.
--                       pipe_out("*a")  - прочитать все, что находится в пайпе.
--                       pipe_out("*l")  - прочитать одну строку.
--                       pipe_out.handle - хэндл пайпа
--
-- pipe_err            - объект ошибок сообщаемых приложением.
--                       pipe_err("*a")  - прочитать все, что находится в пайпе.
--                       pipe_err("*l")  - прочитать одну строку.
--                       pipe_err.handle - хэндл пайпа
--
-- info                - массив с данными о запущенном процессе
-- info.pid            - PID
-- info.tid            - TID
-- info.process_handle - хэндл процесса
-- info.thread_handle  - хэндл потока
--
-- path                - путь для запуска исполняемого файла.
--                       Если путь содержит пробелы, то необходимо
--                       использовать двойные кавычки внутри строки.
--                       Например:
--                       execpipe([["d:\path to my\app.exe"]], nil, "combine")
--
-- env                 - переменные окружения которые должны быть переданы.
--                       Формат стандартный для api: name_1=val_1\0name_N=val_N\0\0
--                       Обратите внимание на два \0\0 завершающих строку.
--                       Если не задан - будут переданы текущие переменные окружения.
--
-- combine             - флаг для объединения pipe_out и pipe_err.
do
    local read_pipe = function(pipe, size)end
    do
        -- ReadFile
        ffi.cdef[[
            typedef unsigned __int64   ULONG_PTR;
            typedef unsigned int       DWORD;
            typedef void*              PVOID;
            typedef PVOID              HANDLE;
            typedef PVOID              HANDLE;
        ]]

        sys.define[[
            typedef struct _OVERLAPPED {
                ULONG_PTR Internal;
                ULONG_PTR InternalHigh;
                union {
                    struct {
                        DWORD Offset;
                        DWORD OffsetHigh;
                    } DUMMYSTRUCTNAME;
                    PVOID Pointer;
                } DUMMYUNIONNAME;
                HANDLE hEvent;
            } OVERLAPPED, *LPOVERLAPPED;
        ]]

        ffi.cdef[[
            typedef int                BOOL;
            typedef void*              PVOID;
            typedef PVOID              HANDLE;
            typedef void*              LPVOID;
            typedef unsigned int       DWORD;
            typedef          DWORD*    LPDWORD;
            BOOL ReadFile(
                HANDLE       hFile,
                LPVOID       lpBuffer,
                DWORD        nNumberOfBytesToRead,
                LPDWORD      lpNumberOfBytesRead,
                LPOVERLAPPED lpOverlapped
            );
        ]]

        -- PeekNamedPipe
        ffi.cdef[[
            typedef int                BOOL;
            typedef void*              PVOID;
            typedef PVOID              HANDLE;
            typedef void*              LPVOID;
            typedef unsigned int       DWORD;
            typedef          DWORD*    LPDWORD;
            BOOL PeekNamedPipe(
                HANDLE  hNamedPipe,
                LPVOID  lpBuffer,
                DWORD   nBufferSize,
                LPDWORD lpBytesRead,
                LPDWORD lpTotalBytesAvail,
                LPDWORD lpBytesLeftThisMessage
            );
        ]]


        local availableBytes = ffi.new("DWORD[1]")
        local buffer_size    = 65536
        local buffer_read    = ffi.new("char[?]", buffer_size)
        local bytesRead      = ffi.new("DWORD[1]")
        read_pipe = function(pipe, size)
            if type(size) == "number" then
                -- Проверяем размер буфера, увеличивем, если нужно.
                if buffer_size < size then
                    buffer_size = 2^math.ceil(math.log(size, 2))
                    buffer_read = ffi.new("char[?]", buffer_size)
                end
                local result = kernel32.ReadFile(pipe, buffer_read, size, bytesRead, nil)
                local lua_buffer = ffi.string(buffer_read, bytesRead[0])
                return lua_buffer
            elseif size == "*a" then
                bytesRead[0] = 0
                kernel32.PeekNamedPipe(pipe, nil, 0, nil, availableBytes, nil)
                if availableBytes[0] > 0 then
                    -- Проверяем размер буфера, увеличивем, если нужно.
                    if buffer_size < availableBytes[0] then
                        buffer_size = 2^math.ceil(math.log(availableBytes[0], 2))
                        buffer_read = ffi.new("char[?]", buffer_size)
                    end
                    local result = kernel32.ReadFile(pipe, buffer_read, availableBytes[0], bytesRead, nil)
                    local lua_buffer = ffi.string(buffer_read, bytesRead[0])
                    return lua_buffer
                else
                    return nil
                end
            elseif size == "*l" or size == nil then
                bytesRead[0] = 0
                kernel32.PeekNamedPipe(pipe, nil, 0, nil, availableBytes, nil)
                if availableBytes[0] > 0 then
                    -- Проверяем размер буфера, увеличивем, если нужно.
                    if buffer_size < availableBytes[0] then
                        buffer_size = 2^math.ceil(math.log(availableBytes[0], 2))
                        buffer_read = ffi.new("char[?]", buffer_size)
                    end
                    -- Получаем данные.
                    kernel32.PeekNamedPipe(pipe, buffer_read, buffer_size, availableBytes, nil, nil)
                    local data = ffi.string(buffer_read, bytesRead[0])
                    -- Находим первый перенос строк.
                    local eol_start, eol_end = data:find("[\r\n]+")
                    if eol_start then
                        -- Удаляем данные из пайпа.
                        local result = kernel32.ReadFile(pipe, buffer_read, eol_end, bytesRead, nil)
                        return data:sub(1, eol_end-1)
                    else -- Переноса нет, возвращаем всю строку.
                        -- Удаляем данные из пайпа.
                        local result = kernel32.ReadFile(pipe, buffer_read, availableBytes[0], bytesRead, nil)
                        return data
                    end
                else
                    return nil
                end
            end
        end
    end


    local write_pipe = function(pipe, data) end
    do
        -- WriteFile
        sys.define[[
            typedef struct _OVERLAPPED {
                ULONG_PTR Internal;
                ULONG_PTR InternalHigh;
                union {
                    struct {
                        DWORD Offset;
                        DWORD OffsetHigh;
                    } DUMMYSTRUCTNAME;
                    PVOID Pointer;
                } DUMMYUNIONNAME;
                HANDLE hEvent;
            } OVERLAPPED, *LPOVERLAPPED;
        ]]

        ffi.cdef[[
            typedef int                BOOL;
            typedef          void*     PVOID;
            typedef          PVOID     HANDLE;
            typedef const    void*     LPCVOID; // must be "CONST"
            typedef unsigned int       DWORD;
            typedef          DWORD*    LPDWORD;
            BOOL WriteFile(
                HANDLE       hFile,
                LPCVOID      lpBuffer,
                DWORD        nNumberOfBytesToWrite,
                LPDWORD      lpNumberOfBytesWritten,
                LPOVERLAPPED lpOverlapped
            );
        ]]
        local buffer_size = 2^15
        local buffer      = ffi.new("char[?]", buffer_size)
        local buffer_lp   = ffi.new("void*"  , buffer)
        local bytesWrote  = ffi.new("DWORD[1]")
        write_pipe = function(pipe, data, size)
            size = size and math.min(#data, size) or #data
            if buffer_size < size then
                buffer_size = 2^math.ceil(math.log(size, 2))
                buffer      = ffi.new("char[?]", buffer_size)
                buffer_lp   = ffi.new("void*"  , buffer)
            end
            ffi.copy(buffer, data, size)
            local result = kernel32.WriteFile(pipe, buffer_lp, size, bytesWrote, nil)
            if result == 0 then
                print("write error. bytesWrote: "..bytesWrote[0].." "..sys.geterr())
                return nil, 0, "ERROR_SUCCESS"
            end
            return true, 0, "ERROR_SUCCESS"
        end
    end

    -- Начало блока для CreateProcessA
    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;
        typedef unsigned short     WORD;
        typedef          BYTE*     LPBYTE; // must be "far"
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
    ]]
    sys.define[[
		typedef struct _STARTUPINFOA {
			DWORD cb;
			LPSTR lpReserved;
			LPSTR lpDesktop;
			LPSTR lpTitle;
			DWORD dwX;
			DWORD dwY;
			DWORD dwXSize;
			DWORD dwYSize;
			DWORD dwXCountChars;
			DWORD dwYCountChars;
			DWORD dwFillAttribute;
			DWORD dwFlags;
			WORD wShowWindow;
			WORD cbReserved2;
			LPBYTE lpReserved2;
			HANDLE hStdInput;
			HANDLE hStdOutput;
			HANDLE hStdError;
		} STARTUPINFOA, *LPSTARTUPINFOA;
    ]]

    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          void*     LPVOID;
        typedef          int       BOOL;
    ]]

    sys.define[[
		typedef struct _SECURITY_ATTRIBUTES {
			DWORD  nLength;
			LPVOID lpSecurityDescriptor;
			BOOL   bInheritHandle;
		} SECURITY_ATTRIBUTES, *PSECURITY_ATTRIBUTES, *LPSECURITY_ATTRIBUTES;
	]]

    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef unsigned int       DWORD;
    ]]
    sys.define[[
		typedef struct _PROCESS_INFORMATION {
			HANDLE hProcess;
			HANDLE hThread;
			DWORD dwProcessId;
			DWORD dwThreadId;
		} PROCESS_INFORMATION, *LPPROCESS_INFORMATION;
	]]

    ffi.cdef[[
        typedef          int       BOOL;
        typedef          char      CHAR;
        typedef const    CHAR*     LPCSTR; // must be "__nullterminated", "CONST"
        typedef          CHAR*     LPSTR;
        typedef unsigned int       DWORD;
        typedef          void*     LPVOID;

        BOOL CreateProcessA(
            LPCSTR lpApplicationName,
            LPSTR lpCommandLine,
            LPSECURITY_ATTRIBUTES lpProcessAttributes,
            LPSECURITY_ATTRIBUTES lpThreadAttributes,
            BOOL bInheritHandles,
            DWORD dwCreationFlags,
            LPVOID lpEnvironment,
            LPCSTR lpCurrentDirectory,
            LPSTARTUPINFOA lpStartupInfo,
            LPPROCESS_INFORMATION lpProcessInformation
        );
    ]]

    -- CreatePipe
    ffi.cdef[[
        typedef unsigned int       DWORD;
        typedef          void*     LPVOID;
        typedef          int       BOOL;
    ]]

    sys.define[[
		typedef struct _SECURITY_ATTRIBUTES {
			DWORD  nLength;
			LPVOID lpSecurityDescriptor;
			BOOL   bInheritHandle;
		} SECURITY_ATTRIBUTES, *PSECURITY_ATTRIBUTES, *LPSECURITY_ATTRIBUTES;
	]]

	ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          HANDLE*   PHANDLE;
        typedef unsigned int       DWORD;
		BOOL CreatePipe(
			PHANDLE hReadPipe,
			PHANDLE hWritePipe,
			LPSECURITY_ATTRIBUTES lpPipeAttributes,
			DWORD nSize
		);
	]]

    -- GetStdHandle
    ffi.cdef[[
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef unsigned int       DWORD;
        HANDLE GetStdHandle(
            DWORD nStdHandle
        );
    ]]

    -- SetStdHandle
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        BOOL SetStdHandle(
            DWORD  nStdHandle,
            HANDLE hHandle
        );
    ]]

    -- CloseHandle
    ffi.cdef[[
        typedef          int       BOOL;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        BOOL CloseHandle(
            HANDLE hObject
        );
    ]]

    -- Максимальный размер строго регламентирован
    -- документацией winapi.
    local cmd_line_buffer = ffi.new("char[32767]")

    -- Создание структуры PROCESS_INFORMATION
    local process_info = ffi.new("PROCESS_INFORMATION")

    local STARTF_USESTDHANDLES = 0x00000001
    local CREATE_NO_WINDOW     = 0x08000000

    -- Создание структуры STARTUPINFO
    local startup_info   = ffi.new("STARTUPINFOA")
    startup_info.cb      = ffi.sizeof(startup_info)
    startup_info.dwFlags = STARTF_USESTDHANDLES

    -- Создание структуры SECURITY_ATTRIBUTES
    local sec_att = ffi.new("SECURITY_ATTRIBUTES")
    sec_att.nLength = ffi.sizeof(sec_att)
    sec_att.lpSecurityDescriptor = nil
    sec_att.bInheritHandle = true

    -- Внимание!
    -- Ввиду очень высокоинтеллектуальных разработок
    -- в Win7 существуют параметры для передачи
    -- пользовательских пайпов, специальный флаг,
    -- который говорит использовать пользовательские пайпы,
    -- но потом они просто игнорируются.
    -- Прям так в доках и написано.
    -- Слишком высокоинтеллектуально.
    -- Мне такое не понять.
    -- Найденный мной способ обойти это сводится к
    -- созданию пользовательских пайпов,
    -- замене стандартных пайпов на пользовательские,
    -- запуску приложения (оно наследует новые стандартные пайпы),
    -- возврату стандартных пайпов на изначальные.
    -- Схема не есть правильная, но альтернатив под win7 я не знаю.
    proc.execpipe = function(path, env, combine)
        local err = nil
        local verbose = nil
        local exe_path = nil
        local cmd_args = nil
        if path:sub(1, 1) == '"' then
            exe_path, cmd_args = path:match([===[^"([^"]+)" *(.*)]===])
        else
            exe_path, cmd_args = path:match([===[^([^ ]+) *(.*)]===])
        end
        local exe_name = exe_path:match([[([^/\]*)$]])

        if cmd_args then
            ffi.copy(cmd_line_buffer, exe_name.." "..cmd_args, #exe_name + #cmd_args+1)
        end

        -- Получаем стандартные хэндлы ввода-вывода.
        local hStdIn  = kernel32.GetStdHandle(-10) -- STD_INPUT_HANDLE
        local hStdOut = kernel32.GetStdHandle(-11) -- STD_OUTPUT_HANDLE
        local hStdErr = kernel32.GetStdHandle(-12) -- STD_ERROR_HANDLE

        local input_pipe_read   = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        local input_pipe_write  = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        kernel32.CreatePipe(input_pipe_read, input_pipe_write, sec_att, 0)

        local output_pipe_read  = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        local output_pipe_write = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        kernel32.CreatePipe(output_pipe_read, output_pipe_write, sec_att, 0)

        local error_pipe_read   = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        local error_pipe_write  = ffi.gc(ffi.new("HANDLE[1]"), kernel32.CloseHandle)
        if combine then
            error_pipe_read  = output_pipe_read
            error_pipe_write = output_pipe_write
        else
            kernel32.CreatePipe(error_pipe_read, error_pipe_write, sec_att, 0)
        end

        -- подмена пайпов
        kernel32.SetStdHandle(-10, input_pipe_read  [0]) -- STD_INPUT_HANDLE
        kernel32.SetStdHandle(-11, output_pipe_write[0]) -- STD_OUTPUT_HANDLE
        kernel32.SetStdHandle(-12, error_pipe_write [0]) -- STD_ERROR_HANDLE

        startup_info.hStdInput  = input_pipe_read  [0]
        startup_info.hStdOutput = output_pipe_write[0]
        startup_info.hStdError  = error_pipe_write [0]

        -- Передаем переменные окружения.
        env = env or proc.getenvironment("string")
        local env_ptr = nil
        if env then
            env_ptr = ffi.new("char[?]", #env)
            ffi.copy(env_ptr, env, #env)
        else
            env_ptr = ffi.new("char[?]", #env)
            ffi.copy(env_ptr, env, #env)
        end

        -- Запуск процесса                                                                v  - deattach 8
        local result = kernel32.CreateProcessA(exe_path, cmd_line_buffer, nil, nil, true, 0, env_ptr, nil, startup_info, process_info)
        err, verbose = sys.geterr()

        if result == 0 then
            err, verbose = sys.geterr()
            print("Process starting error: "..err.." "..verbose)
        end

        -- Возвращаем стандартные пайпы на место
        kernel32.SetStdHandle(-10, hStdIn)  -- STD_INPUT_HANDLE
        kernel32.SetStdHandle(-11, hStdOut) -- STD_OUTPUT_HANDLE
        kernel32.SetStdHandle(-12, hStdErr) -- STD_ERROR_HANDLE

        if process_info.dwProcessId == 0 then
            print(path.." process start failed. Error: "..err.." "..verbose)
        end

        local in_obj        = {}
        in_obj.handle       = ffi.gc(ffi.new("HANDLE", input_pipe_write[0]), kernel32.CloseHandle)
        local input_obj_mt  = {}
        input_obj_mt.__call = function(self, size)
            return write_pipe(in_obj.handle, size)
        end
        input_obj_mt.__tostring = function(self)
            return tostring(in_obj.handle)
        end
        setmetatable(in_obj, input_obj_mt)

        local out_obj     = {}
        out_obj.handle    = ffi.gc(ffi.new("HANDLE", output_pipe_read[0]), kernel32.CloseHandle)
        local out_obj_mt  = {}
        out_obj_mt.__call = function(self, size)
            return read_pipe(out_obj.handle, size)
        end
        out_obj_mt.__tostring = function(self)
            return tostring(out_obj.handle)
        end
        setmetatable(out_obj, out_obj_mt)

        local err_obj     = {}
        err_obj.handle    = ffi.gc(ffi.new("HANDLE", error_pipe_read[0]), kernel32.CloseHandle)
        local err_obj_mt  = {}
        err_obj_mt.__call = function(self, size)
            return read_pipe(err_obj.handle, size)
        end
        err_obj_mt.__tostring = function(self)
            return tostring(err_obj.handle)
        end
        setmetatable(err_obj, err_obj_mt)

        local info = {}
        info.pid            = process_info.dwProcessId
        info.tid            = process_info.dwThreadId
        info.process_handle = process_info.hProcess
        info.thread_handle  = process_info.hThread
        return in_obj, out_obj, err_obj, info
    end
end


-- path, err, verbose = exepath([handle|pid])
-- Получает путь к исполняемому файлу.
--
-- path          - массив с результатом
-- path.exe      - имя файла
-- path.path     - путь к каталогу с файлом
-- path.fullpath - полный путь к exe
-- Пример:
-- path.exe      - cmd.exe
-- path.path     - c:\windows\system32\
-- path.fullpath - c:\windows\system32\cmd.exe
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки.
--
-- handle - handle процесса информацию о котором получаем.
--          Значение по умолчанию window.
--
-- pid    - pid процесса информацию о котором получаем.
--
-- Внимание! Указывается либо pid либо handle.
-- Тип данный pid - number, handle - HWND (void*).
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          DWORD*    PDWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          char      CHAR;
        typedef          CHAR*     LPSTR;

        BOOL QueryFullProcessImageNameA(
            HANDLE hProcess,
            DWORD  dwFlags,
            LPSTR  lpExeName,
            PDWORD lpdwSize
        );

        HANDLE OpenProcess(
            DWORD dwDesiredAccess,
            BOOL  bInheritHandle,
            DWORD dwProcessId);

        BOOL CloseHandle(HANDLE hObject);
        DWORD GetLastError();
    ]]
    local PROCESS_QUERY_INFORMATION = 0x0400

    -- В связи с тем, что QueryFullProcessImageNameA
    -- перезаписывает переданный ему размер буфера,
    -- То приходится хранить переменную с реальным
    -- размером буфера и каждый раз записывать
    -- в массив значение переменной под убой.
    -- Тут видимо был новый рецепт наркоты
    -- и конкретный передоз. Браво майкрософт!
    -- При том, что QueryFullProcessImageNameA
    -- возвращает BOOL, который по факту так же
    -- занимает тот же размер что и значение
    -- размера буфера в современных реалиях,
    -- очень много вопросов вызвает в чем была проблема
    -- вместо BOOL возвращать количество копируемых
    -- символов, так же как и других функциях.

    local buffer_size     = 32768
    local buffer          = ffi.new("char[?]", buffer_size)
    local buffer_size_ptr = ffi.new("DWORD[1]")

    proc.exepath = function(pid)
        pid = type(pid) == "number" and pid or wnd.windowpid(pid)

        local process_handle = kernel32.OpenProcess(
            PROCESS_QUERY_INFORMATION, 0, pid)
        if process_handle == 0 then return nil, sys.geterr() end

        repeat
            buffer_size_ptr[0] = buffer_size
            local name_result = kernel32.QueryFullProcessImageNameA(
            process_handle, 0, buffer, buffer_size_ptr)
            -- Завершилось неудачей?
            -- 122 ошибка - нехватка буфера,
            -- при ее возникновении увеличиваем
            -- размер буфера, нет смысла прерывть функцию.
            local err, verbose = sys.geterr()
            if name_result == 0 and err ~= 122 then
                return nil, err, verbose
            end

            -- +1 на ноль в конце строки
            local lenght = buffer_size_ptr[0] + 1
            -- Увеличиваем буфер, если необходимо
            if lenght > buffer_size then
                buffer_size = buffer_size*2
                buffer = ffi.new("char[?]", buffer_size)
            end
        until name_result ~= 0


        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, sys.geterr() end

        local result    = {}
        result.fullpath = ffi.string(buffer)
        result.path, result.exe = result.fullpath:match("(.*\\)([^\\]*)")
        return result, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = terminate(victim, name, class, pos, pid)
-- Форсированно завершает процесс.
--
-- result  - успех выполнения true/nil.
--
-- err     - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- вариант 1
-- victim  - можеть быть задан хэндлом тип HWND (cdata void*)
--           name, class, pos, pid - игнорируются.
--
-- вариант 2
--           Любой из параметров может быть nil,
--           для исключения его из фильтрации.
--
-- victim  - путь к исполняемому файлу (exe).
--           Задается, как паттерн для regexp.
--           Проверка происходит в полном пути.
--
-- name    - заголовок окна. Задается как паттерн
--           для regexp.
--
-- class   - имя класса окна. Задается как паттерн
--           для regexp.
--
-- pos            - таблица с положением и размером окна.
-- pos.min_x      - начальная X координата области поиска
-- pos.min_y      - начальная Y координата области поиска
-- pos.min_width  - минимальная ширина окна
-- pos.min_height - минимальная высота окна
-- pos.max_x      - конечная  X координата области поиска
-- pos.max_y      - конечная  Y координата области поиска
-- pos.max_width  - максимальная ширина окна
-- pos.max_height - максимальная высота окна
--
-- pid     - pid процесса.
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef unsigned int       UINT;
        typedef          void*     HWND;
        typedef          DWORD*    LPDWORD;
        DWORD GetWindowThreadProcessId(HWND hwnd, LPDWORD lpdwProcessId);
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        DWORD GetLastError();
        BOOL TerminateProcess(HANDLE hProcess, UINT uExitCode);
        BOOL CloseHandle(HANDLE hObject);
    ]]
    local PROCESS_TERMINATE = 0x0001
    proc.terminate = function(victim, name, class, pos, pid)
        local pid_to_kill = {}
        if type(victim) == "cdata" then
            local _        = nil
            local err      = nil
            local verbose  = nil
            pid_to_kill[1] = {}
            pid_to_kill[1].pid, _, err, verbose = wnd.windowpid(victim)
            if err ~= 0 then return nil, err, verbose end
        elseif pid then
            pid_to_kill[1] = {}
            pid_to_kill[1].pid = pid
        else
            local result, err, verbose = wnd.findwindow(victim, name, class, pos, pid)
            if result then
                pid_to_kill = result
            else
                return nil, err, verbose
            end
        end

        for i = 1, #pid_to_kill do
            local process_handle   = kernel32.OpenProcess(
                PROCESS_TERMINATE, 0, pid_to_kill[i].pid)
            if process_handle     == 0 then return nil, sys.geterr() end
            local terminate_result = kernel32.TerminateProcess(process_handle, 0)
            if terminate_result   == 0 then return nil, sys.geterr() end
            local close_result     = kernel32.CloseHandle(process_handle)
            if close_result       == 0 then return nil, sys.geterr() end
        end
        return true, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = setpriority(prio, [handle|pid])
-- Устанавливает приоритет для цли.
--
-- result - успех выполнения true/false.
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- prio   - желаемый приоритет. Возможные варианты:
--          "real"   - реального времени
--          "high"   - высокий
--          "above"  - выше среднего
--          "normal" - средний
--          "below"  - ниже среднего
--          "idle"   - низкий
--
-- handle - хэндл процесса HWND (void*)
--          По умолчанию window.
--
-- pid    - pid процесса (number).
do
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        BOOL SetPriorityClass(
            HANDLE hProcess,
            DWORD  dwPriorityClass
        );
        DWORD GetLastError();
    ]]
    local PROCESS_SET_INFORMATION = 0x0200
    local prio_list = {}
    prio_list.real   = 0x00000100 -- REALTIME_PRIORITY_CLASS
    prio_list.high   = 0x00000080 -- HIGH_PRIORITY_CLASS
    prio_list.above  = 0x00008000 -- ABOVE_NORMAL_PRIORITY_CLASS
    prio_list.normal = 0x00000020 -- NORMAL_PRIORITY_CLASS
    prio_list.below  = 0x00004000 -- BELOW_NORMAL_PRIORITY_CLASS
    prio_list.idle   = 0x00000040 -- IDLE_PRIORITY_CLASS

    proc.setpriority = function(prio, target)
        target = target or window
        if type(target) == "cdata" then
            target = wnd.windowpid(target)
        end

        local process_handle = kernel32.OpenProcess(PROCESS_SET_INFORMATION, 0, target)
        if process_handle   == 0 then return nil, sys.geterr() end
        local prio_result    = kernel32.SetPriorityClass(process_handle, prio_list[prio])
        if prio_result      == 0 then return nil, sys.geterr() end
        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, sys.geterr() end
        return true, 0, "ERROR_SUCCESS"
    end
end


-- result, err, verbose = setaffinity(mask, [handle|pid])
-- Устанавливает приоритет для цли.
--
-- result - успех выполнения true/false.
--
-- err    - код ошибки. 0 - успех.
--
-- verbose - текстовая расшифровка ошибки
--
-- mask   - битовая маска соответствия с процессорами.
--          Может быть задана числом в виде
--          number или ULONG_PTR.
--          Внимание number может корректно хранить
--          только 52 бита, т.е. 52 процессора.
--          Альтернативный способ задать маску
--          строка с символами "Y", где
--          позиция = номер процессора.
--          Прочие символы игнорируются.
--          Пример:
--          "01YY456Y" - задаст соответствие
--          2, 3, 7 процессорам (отсчет с нуля).
--          Цифры используются только для удобства
--          в качестве разделителя.
--
-- handle - хэндл процесса HWND (void*)
--          По умолчанию window.
--
-- pid    - pid процесса (number).
do
    if ffi.arch == "x64" then
        ffi.cdef[[
                typedef unsigned __int64 ULONG_PTR;
            ]]
    else
        ffi.cdef[[
                typedef unsigned long ULONG_PTR;
            ]]
    end
    ffi.cdef[[
        typedef          int       BOOL;
        typedef unsigned int       DWORD;
        typedef          void*     PVOID;
        typedef          PVOID     HANDLE;
        typedef          ULONG_PTR DWORD_PTR;
        HANDLE OpenProcess(DWORD dwDesiredAccess,
                           BOOL  bInheritHandle,
                           DWORD dwProcessId);
        BOOL SetProcessAffinityMask(
            HANDLE    hProcess,
            DWORD_PTR dwProcessAffinityMask
        );
        DWORD GetLastError();
    ]]
    local PROCESS_SET_INFORMATION = 0x0200
    local mask = ffi.new("ULONG_PTR[1]")

    -- mask объявлено выше т.к. завист от битности.
    local parse_mask = function(str)
        for i = 1, #str do
            if str:sub(i, i) == "Y" then
                mask = mask + 2^i
            end
        end
        return mask
    end

    proc.setaffinity = function(new_mask, target)
        if type(new_mask) == "string" then
            mask[0] = parse_mask(new_mask)
        else
            mask[0] = new_mask
        end

        target = target or window
        if type(target) == "cdata" then
            target = wnd.windowpid(target)
        end

        local process_handle = kernel32.OpenProcess(PROCESS_SET_INFORMATION, 0, target)
        if process_handle   == 0 then return nil, sys.geterr() end
        local aff_result     = kernel32.SetProcessAffinityMask(process_handle, mask[0])
        if aff_result       == 0 then return nil, sys.geterr() end
        local close_result   = kernel32.CloseHandle(process_handle)
        if close_result     == 0 then return nil, sys.geterr() end
        return true, 0, "ERROR_SUCCESS"
    end
end

return proc


