--[[
==============================================================================
 Tesseract & Leptonica OCR-  UOPilot (LuaJIT)
==============================================================================

 :
       Tesseract 5.5.1 
 Leptonica 1.86,     UOPilot  LuaJIT.
      (OCR)   ,
          .

 :
 *   : "auto" (,   )  "manual"
   (    ).
 *     Tesseract (PSM)   (OEM).
 *    ,     .
 *    Leptonica: ,  ,
    ,  ,  
    (  Simple, Otsu  Sauvola).
 *      .

 :
 1.  DLL (    UOPilot):
    * tesseract55.dll
    * leptonica1860.dll
 2.  DLL (    ):
    * gif.dll, jpeg62.dll, liblzma.dll, libpng16.dll, libsharpyuv.dll,
      libwebp.dll, libwebpmux.dll, openjp2.dll, tiff.dll, zlib1.dll
 3.  win-125x.lua     .
 4.  'tessdata'   .

   'TESSDATA':
 1.     'tessdata_fast'      .
 2.        :
    tessdata: https://github.com/tesseract-ocr/tessdata
    tessdata_best: https://github.com/tesseract-ocr/tessdata_best
    tessdata_fast: https://github.com/tesseract-ocr/tessdata_fast
 3.      'tess.data_path = 'tessdata_fast''.
 4.   ,    **      (. ).

     :
 --------------------------------------------------------------------------
 --lua
 --  
 local tess = require('tesseract')

 -- []     
 tess.data_path = 'tessdata_fast'

 -- []   
 tess.options.page_seg_mode = "6"
 tess.options.preprocessing_mode = "auto"
 tess.options.tesseract_invert = true

 --  OCR
 local image_address, width, height, length = getimage(0, 0, 100, 50, 'abs')
 if image_address ~= 0 then
     local results = tess.ocr(image_address, width, height, length)
     deleteimage(image_address)
     log(results)
 end
 --------------------------------------------------------------------------

 :
 Madeus

 :
 MIT License

 Copyright (c) 2024 Madeus

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all
 copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 SOFTWARE.

==============================================================================
]]

local ffi = require('ffi')
require('win-125x')

--  
local tesseract = ffi.load('tesseract55')
local lept = ffi.load('leptonica1860')

-- 
local CONSTANTS = {
    RIL_BLOCK = 0,
    RIL_PARA = 1,
    RIL_TEXTLINE = 2,
    RIL_WORD = 3,
    RIL_SYMBOL = 4
}

--  FFI  Leptonica
ffi.cdef[[
    /* Core Types */
    typedef struct Pix Pix;

    /* Create/Destroy */
    Pix* pixCreate(int width, int height, int depth);
    void pixDestroy(Pix** pix);

    /* Accessors */
    int pixGetWidth(Pix* pix);
    int pixGetHeight(Pix* pix);
    int pixGetDepth(Pix* pix);
    unsigned char* pixGetData(Pix* pix);
    int pixGetWpl(Pix* pix);

    /* Color Conversion */
    Pix* pixConvertRGBToGray(Pix* pixs, float rwt, float gwt, float bwt);
    Pix* pixConvertTo8(Pix* pixs);

    /* Filters */
    Pix* pixScale(Pix* pixs, float scalex, float scaley);
    Pix* pixEqualizeTRC(Pix* pixs, float fract, int factor);
    Pix* pixRankFilter(Pix* pixs, int wc, int hc, float rank);
    Pix* pixUnsharpMaskingGray(Pix* pixs, int size, float fract);
    Pix* pixInvert(Pix* pixs, Pix* pixd);

    /* Binarization */
    Pix* pixThresholdToBinary(Pix* pixs, int thresh);
    Pix* pixOtsuAdaptiveThreshold(Pix* pixs, int sx, int sy, int smoothx, int smoothy, float scorefract, Pix** ppixth, Pix** ppixd);
    Pix* pixSauvolaBinarizeTiled(Pix* pixs, int whsize, float factor, int nx, int ny, Pix** ppixth, Pix** ppixd);

    /* I/O */
    void pixWrite(const char* filename, Pix* pix, int format);
]]

--  FFI  Tesseract
ffi.cdef[[
    /* Core Types */
    typedef void TessBaseAPI;

    /* API Lifecycle */
    TessBaseAPI* TessBaseAPICreate(void);
    int TessBaseAPIInit3(TessBaseAPI* handle, const char* datapath, const char* language);
    void TessBaseAPIClear(TessBaseAPI* handle);
    void TessBaseAPIDelete(TessBaseAPI* handle);

    /* Image Handling */
    void TessBaseAPISetImage(TessBaseAPI* handle, const unsigned char* imagedata, int width, int height, int bytes_per_pixel, int bytes_per_line);
    void TessBaseAPISetImage2(TessBaseAPI* handle, Pix* pix);
    int TessBaseAPISetVariable(TessBaseAPI* handle, const char* name, const char* value);

    /* Recognition */
    int TessBaseAPIRecognize(TessBaseAPI* handle, void* monitor);
    char* TessBaseAPIGetUTF8Text(TessBaseAPI* handle);

    /* Iterator (for coordinates) */
    typedef struct TessResultIterator TessResultIterator;
    typedef struct TessPageIterator TessPageIterator;
    TessResultIterator* TessBaseAPIGetIterator(TessBaseAPI* handle);
    TessPageIterator* TessResultIteratorGetPageIterator(TessResultIterator* handle);
    bool TessPageIteratorBoundingBox(TessPageIterator* handle, int level, int* left, int* top, int* right, int* bottom);
    char* TessResultIteratorGetUTF8Text(TessResultIterator* handle, int level);
    bool TessPageIteratorNext(TessPageIterator* handle, int level);
    void TessResultIteratorDelete(TessResultIterator* handle);
]]


--   
local function check(condition, message, ...)
    if not condition then
        error(string.format(message, ...), 2)
    end
end

--     
local valid_values = {
    page_seg_mode = { min = 0, max = 13 },
    ocr_engine_mode = { min = 0, max = 3 },
    recognition_level = { "block", "para", "textline", "word", "symbol" },
    preprocessing_mode = { "auto", "manual" },
    dotproduct = { "auto", "generic", "native", "avx", "sse" },
    binary_method = { "simple", "otsu", "sauvola" }
}

--    
local function check_param(name, value)
    local spec = valid_values[name]
    if type(spec) == "table" and spec.min then
        local num = tonumber(value)
        check(num ~= nil and num >= spec.min and num <= spec.max,
            "  %s: %s.  : %d-%d", name, value, spec.min, spec.max)
    elseif type(spec) == "table" then
        local found = false
        for _, v in ipairs(spec) do
            if v == value then found = true break end
        end
        check(found, "  %s: %s.  : %s",
            name, value, table.concat(spec, ", "))
    else
        error(" : " .. name)
    end
end

--   
local tess = {}
tess.data_path = 'tessdata_fast'
tess.language = 'eng+rus'

--[[
==============================================================================
   (tess.options)
==============================================================================
     Tesseract   Leptonica.
  ,    OCR.
]]
tess.options = {
    --
    -- 1.   OCR
    --

    -- (string)  .
    -- * "auto":  . Tesseract   24- .  ,     .
    -- * "manual":  .     Leptonica (to_gray, scale, median, binary  ..).
    preprocessing_mode = "auto",

    -- (boolean)   .
    -- * true:   { {text="...", left=0, top=0, right=0, bottom=0}, ... }
    -- * false:      .
    return_coordinates = false,

    -- (string)    'return_coordinates = true'.
    --  , **     .
    --    'get_text_only'.
    -- * "block":     .
    -- * "para":   .
    -- * "textline":    (   ).
    -- * "word":   -   ().
    -- * "symbol":   -  .
    recognition_level = "word",

    -- (boolean)    (  'tess.postprocess_text').
    --      OCR  .
    -- : tess.postprocess_text = { {"%s+", " "}, {"Il", "ll"} }
    -- (       "Il"  "ll")
    use_postprocess = false,

    --
    -- 2.   TESSERACT
    --

    -- (string)    (PSM). (0-13)
    -- * "0":      (OSD).
    -- * "1":     OSD.
    -- * "2":  ,   OSD  OCR.
    -- * "3":   ,   OSD ().
    -- * "4":     .
    -- * "5":     .
    -- * "6":    .
    -- * "7":     .
    -- * "8":    .
    -- * "9":      .
    -- * "10":    .
    -- * "11":   (      ).
    -- * "12":    OSD.
    -- * "13": ""  (  ,  . ).
    page_seg_mode = "3",

    -- (string)  OCR- (OEM). (0-3)
    -- * "0":  Tesseract ( "Legacy" ).
    -- * "1":   (LSTM) (  Tesseract 5).
    -- * "2": Tesseract (Legacy) +  (LSTM) ().
    -- * "3":   (  ,  LSTM/1).
    ocr_engine_mode = "1",

    -- (string / false)  . Tesseract   **  .
    -- : "0123456789" ( )
    whitelist = false,

    -- (string / false)  . Tesseract    .
    -- : ".,!?" ( )
    blacklist = false,

    -- (boolean)     .
    -- * true: "Hello    World"  "Hello    World" (  4 ).
    -- * false: "Hello    World"  "Hello World" ( ).
    preserve_spaces = false,

    -- (boolean)   Tesseract.       "auto".
    --   ,  'invert = true'   "manual".
    tesseract_invert = false,

    -- (number)  DPI (  )  .
    -- Tesseract    DPI >= 300.  nil,   .
    dpi = nil,

    -- (string)  . "auto" -  .
    -- "auto", "generic", "native", "avx", "sse".
    dotproduct = "auto",

    -- (number) ID . ,     Tesseract
    --    .  'nil' ().
    font_id = nil,

    --
    -- 3.    ( preprocessing_mode = "manual")
    --

    -- (table)   .
    -- : 'to_gray'   ,    .
    filter_order = {"scale_up", "equalize_hist", "median", "unsharp_mask", "invert", "binary"},

    -- (boolean)  .
    scale_up = false,

    -- (number)  . 1.0 =  .
    --  (, 2.0)     .
    scale_factor = 1.0,

    -- (boolean)   .
    -- "" .    ,  
    -- , , -.   
    --   .
    equalize_hist = false,

    -- (boolean)   .
    --    "  " ( ).
    median = false,

    -- (number)  ""    (, >= 3).
    -- 3 = , 5 =   .
    median_size = 3,

    -- (boolean)     (Unsharp Masking).
    --      .
    unsharp_mask = false,

    -- (number)  ""    (, >= 3).
    unsharp_size = 5,

    -- (float)    (0.0 - 1.0).
    -- 0.5 = .
    unsharp_fract = 0.5,

    -- (boolean)   ( <-> ).
    -- Tesseract  **   ** .
    --       ,   (invert = true).
    invert = false,

    -- (boolean)   (  /).
    binary = false,

    -- (string)  .
    -- * "simple":   ( 'binary_threshold').
    -- * "otsu":   .
    -- * "sauvola":   .
    --
    -- : "simple" -        
    -- (    ). "otsu"  "sauvola" - 
    --   ,    .
    binary_method = "simple",

    -- (number)   'binary_method = "simple"' (0-255).
    binary_threshold = 128,

    -- ---   Otsu ('binary_method = "otsu"') ---
    -- (number)  ""  X   (>= 16).
    -- Leptonica          .
    otsu_tile_x = 32,
    -- (number)  ""  Y   (>= 16).
    otsu_tile_y = 32,
    -- (number)     X (0 = , 2 = ).
    --   ""  .
    otsu_smooth_x = 2,
    -- (number)     Y (0 = , 2 = ).
    otsu_smooth_y = 2,
    -- (float)   (0.0 - 1.0). 0.1 - . (  ).
    otsu_score_fract = 0.1,

    -- ---   Sauvola ('binary_method = "sauvola"') ---
    -- (number)  ""  (, >= 15).
    --          .
    sauvola_whsize = 15,
    -- (float)  (0.0 - 1.0). 0.35 - .
    --  (. 0.5)   ,  - .
    sauvola_factor = 0.35,
    -- (number)    X,     (1 =  ).
    sauvola_nx = 1,
    -- (number)    Y (1 =  ).
    sauvola_ny = 1,

    --
    -- 4.  
    --

    -- (boolean)   -   UOPilot.
    debug = false,

    -- (boolean)    (to_gray.bmp, binary.bmp  ..)
    --    UOPilot.
    save_intermediate_images = false,
}

--   
tess.postprocess_text = {}

--    
local function check_scale_factor(scale_factor)
    check(scale_factor > 0, "  scale_factor: %s.  : > 0", scale_factor)
end

local function check_font_id(font_id)
    if font_id then
        local num = tonumber(font_id)
        check(num and num >= 0, "  font_id: %s.  : >= 0", font_id)
    end
end

local function check_filter_params(options)
    if options.binary then
        check_param("binary_method", options.binary_method)
        if options.binary_method == "otsu" then
            check(options.otsu_tile_x >= 16, "otsu_tile_x   >= 16")
            check(options.otsu_tile_y >= 16, "otsu_tile_y   >= 16")
            check(options.otsu_smooth_x >= 0, "otsu_smooth_x   >= 0")
            check(options.otsu_smooth_y >= 0, "otsu_smooth_y   >= 0")
            check(options.otsu_score_fract >= 0 and options.otsu_score_fract <= 1.0, "otsu_score_fract   0.0-1.0")
        elseif options.binary_method == "sauvola" then
            check(options.sauvola_whsize % 2 == 1 and options.sauvola_whsize >= 15, "sauvola_whsize     >= 15")
            check(options.sauvola_factor >= 0 and options.sauvola_factor <= 1.0, "sauvola_factor   0.0-1.0")
            check(options.sauvola_nx >= 1, "sauvola_nx   >= 1")
            check(options.sauvola_ny >= 1, "sauvola_ny   >= 1")
        else -- "simple"
            check(options.binary_threshold >= 0 and options.binary_threshold <= 255,
                "  binary_threshold: %s.  : 0-255", options.binary_threshold)
        end
    end
    if options.median then
        check(options.median_size % 2 == 1 and options.median_size >= 3,
            "median_size     >= 3,  : %s", options.median_size)
    end
    if options.unsharp_mask then
        check(options.unsharp_size % 2 == 1 and options.unsharp_size >= 3,
            "unsharp_size     >= 3,  : %s", options.unsharp_size)
        check(options.unsharp_fract >= 0 and options.unsharp_fract <= 1.0, "unsharp_fract   0.0-1.0")
    end
end

--   
local function save_pix(pix, filename)
    local width = lept.pixGetWidth(pix)
    local height = lept.pixGetHeight(pix)
    local depth = lept.pixGetDepth(pix)
    local data = lept.pixGetData(pix)
    local wpl = lept.pixGetWpl(pix)

    if tess.options.debug then
        log(string.format(" Pix: width=%d, height=%d, depth=%d, wpl=%d, data=%s", width, height, depth, wpl, tostring(data ~= nil)))
    end

    check(width > 0 and height > 0, "  Pix: width=%d, height=%d", width, height)
    check(data ~= nil, " Pix  NULL")

    if tess.options.debug and tess.options.save_intermediate_images then
        lept.pixWrite(filename, pix, 1) -- 1 = IFF_BMP
        log("Pix   " .. filename)
    end
end

--  
local function apply_postprocess(text, replacements)
    for _, replacement in ipairs(replacements) do
        local from, to = replacement[1], replacement[2]
        text = text:gsub(from, to)
    end
    return text
end

--    
local function cleanup(api, pixs, initial_pix)
    if tess.options.debug then log(" API   Pix") end
    if api then
        tesseract.TessBaseAPIClear(api)
        tesseract.TessBaseAPIDelete(api)
        if tess.options.debug then log(" API Tesseract") end
    end
    if pixs then
        for i, pix in ipairs(pixs) do
            if pix and ffi.cast("Pix*", pix) ~= nil then
                lept.pixDestroy(ffi.new("Pix*[1]", pix))
                if tess.options.debug then log(" Pix ( " .. i .. ")") end
            end
        end
    end
    if initial_pix and ffi.cast("Pix*", initial_pix) ~= nil then
        lept.pixDestroy(ffi.new("Pix*[1]", initial_pix))
        if tess.options.debug then log("  Pix") end
    end
end

--    
local function get_text_only(api)
    if tess.options.debug then log("  ") end
    local text_ptr = tesseract.TessBaseAPIGetUTF8Text(api)
    check(text_ptr ~= nil, "TessBaseAPIGetUTF8Text()  NULL")

    local raw_result = ffi.string(text_ptr)
    if tess.options.debug then log(" : " .. raw_result) end

    local result = utf8_to_win(raw_result)
    if tess.options.debug then log(" : " .. result) end

    if tess.options.use_postprocess and #tess.postprocess_text > 0 then
        result = apply_postprocess(result, tess.postprocess_text)
        if tess.options.debug then log("  : " .. result) end
    end

    return result
end

local function get_text_with_coordinates(api)
    if tess.options.debug then log("   ") end
    local recognize_result = tesseract.TessBaseAPIRecognize(api, nil)
    if tess.options.debug then log(" Recognize: " .. tostring(recognize_result)) end
    check(recognize_result == 0, "TessBaseAPIRecognize()    !  : %d", recognize_result)

    local iterator = tesseract.TessBaseAPIGetIterator(api)
    if tess.options.debug then log(": " .. tostring(iterator ~= nil)) end
    check(iterator ~= nil, "TessBaseAPIGetIterator()  NULL")

    local page_iterator = tesseract.TessResultIteratorGetPageIterator(iterator)
    if tess.options.debug then log(" : " .. tostring(page_iterator ~= nil)) end
    check(page_iterator ~= nil, "TessResultIteratorGetPageIterator()  NULL")

    local level
    check_param("recognition_level", tess.options.recognition_level)
    if tess.options.recognition_level == "block" then
        level = CONSTANTS.RIL_BLOCK
    elseif tess.options.recognition_level == "para" then
        level = CONSTANTS.RIL_PARA
    elseif tess.options.recognition_level == "textline" then
        level = CONSTANTS.RIL_TEXTLINE
    elseif tess.options.recognition_level == "word" then
        level = CONSTANTS.RIL_WORD
    elseif tess.options.recognition_level == "symbol" then
        level = CONSTANTS.RIL_SYMBOL
    end

    local results = {}
    local left = ffi.new("int[1]")
    local top = ffi.new("int[1]")
    local right = ffi.new("int[1]")
    local bottom = ffi.new("int[1]")

    repeat
        local text_ptr = tesseract.TessResultIteratorGetUTF8Text(iterator, level)
        if text_ptr ~= nil then
            local raw_text = ffi.string(text_ptr)
            local text = utf8_to_win(raw_text)
            if tess.options.debug then log(" : " .. text) end

            if tess.options.use_postprocess and #tess.postprocess_text > 0 then
                text = apply_postprocess(text, tess.postprocess_text)
            end

            local has_box = tesseract.TessPageIteratorBoundingBox(page_iterator, level, left, top, right, bottom)
            if has_box then
                table.insert(results, {
                        text = text,
                        left = left[0],
                        top = top[0],
                        right = right[0],
                        bottom = bottom[0]
                    })
            end
        end
    until not tesseract.TessPageIteratorNext(page_iterator, level)

    tesseract.TessResultIteratorDelete(iterator)
    if tess.options.debug then log(" : " .. #results) end
    return results
end

--    Tesseract API
local function get_tesseract_api(data_path, language)
    if tess.options.debug then log(" API Tesseract, data_path: " .. tostring(data_path) .. " : " .. tostring(language)) end
    local api = tesseract.TessBaseAPICreate()
    local data_path_c = data_path and ffi.cast("const char*", data_path) or nil
    local language_c = language and ffi.cast("const char*", language) or nil
    local init_result = tesseract.TessBaseAPIInit3(api, data_path_c, language_c)
    if tess.options.debug then log("  Tesseract: " .. tostring(init_result)) end
    check(init_result == 0, "  Tesseract: %d", init_result)
    return api
end

--     tess.options  Tesseract API
local function apply_tesseract_options(api)
    if tess.options.debug then log("  Tesseract") end

    check_param("page_seg_mode", tess.options.page_seg_mode)
    local set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_pageseg_mode", tostring(tess.options.page_seg_mode))
    check(set_result == 1, "    Tesseract: tessedit_pageseg_mode")

    if tess.options.blacklist then
        set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_char_blacklist", win_to_utf8(tess.options.blacklist))
        check(set_result == 1, "    Tesseract: tessedit_char_blacklist")
    end

    if tess.options.whitelist then
        set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_char_whitelist", win_to_utf8(tess.options.whitelist))
        check(set_result == 1, "    Tesseract: tessedit_char_whitelist")
    end

    check_param("ocr_engine_mode", tess.options.ocr_engine_mode)
    set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_ocr_engine_mode", tostring(tess.options.ocr_engine_mode))
    check(set_result == 1, "    Tesseract: tessedit_ocr_engine_mode")

    if tess.options.preserve_spaces then
        set_result = tesseract.TessBaseAPISetVariable(api, "preserve_interword_spaces", "1")
        check(set_result == 1, "    Tesseract: preserve_interword_spaces")
    end

    set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_do_invert", tess.options.tesseract_invert and "1" or "0")
    check(set_result == 1, "    Tesseract: tessedit_do_invert")

    check_font_id(tess.options.font_id)
    if tess.options.font_id then
        set_result = tesseract.TessBaseAPISetVariable(api, "tessedit_font_id", tostring(tess.options.font_id))
        check(set_result == 1, "    Tesseract: tessedit_font_id")
    end

    check_param("dotproduct", tess.options.dotproduct)
    set_result = tesseract.TessBaseAPISetVariable(api, "dotproduct", tess.options.dotproduct)
    check(set_result == 1, "    Tesseract: dotproduct")

    if tess.options.dpi then
        check(type(tess.options.dpi) == "number" and tess.options.dpi > 0,
            "  dpi: %s.  :  > 0", tess.options.dpi)
        set_result = tesseract.TessBaseAPISetVariable(api, "user_defined_dpi", tostring(tess.options.dpi))
        check(set_result == 1, "    Tesseract: user_defined_dpi")
    end
end

--    Tesseract ( "auto")
local function run_tesseract(image_data, width, height, channels, width_step, data_path, language)
    if tess.options.debug then
        log(" Tesseract ( auto), width: " .. tostring(width) .. " height: " .. tostring(height) ..
            " channels: " .. tostring(channels) .. " width_step: " .. tostring(width_step))
    end
    local api = get_tesseract_api(data_path, language)

    local image_data_ptr = ffi.cast('unsigned char*', image_data)
    if tess.options.debug then log("   : " .. tostring(image_data_ptr ~= nil)) end
    check(image_data_ptr ~= nil, "     NULL") --  

    tesseract.TessBaseAPISetImage(api, image_data, width, height, channels, width_step)

    apply_tesseract_options(api)

    local results = tess.options.return_coordinates and get_text_with_coordinates(api) or get_text_only(api)
    if tess.options.debug then log("  Tesseract: " .. type(results)) end
    return results, api
end

--    Tesseract   Pix ( "manual")
local function run_tesseract_pix(pix, data_path, language)
    if tess.options.debug then
        log(" Tesseract ( manual  Pix), data_path: " .. tostring(data_path) .. " : " .. tostring(language))
    end
    local api = get_tesseract_api(data_path, language)

    check(pix ~= nil, "Pix object is NULL in run_tesseract_pix")
    tesseract.TessBaseAPISetImage2(api, pix)

    apply_tesseract_options(api)

    local results = tess.options.return_coordinates and get_text_with_coordinates(api) or get_text_only(api)
    if tess.options.debug then log("  Tesseract (Pix): " .. type(results)) end
    return results, api
end


--    Leptonica
local function filter_to_gray(pix, options, temp_pixs)
    if options.debug then log("    ") end

    local pix_gray = lept.pixConvertRGBToGray(pix, 0.299, 0.587, 0.114)
    if pix_gray == nil then
        if options.debug then log("pixConvertRGBToGray   (  24bpp),  pixConvertTo8") end
        pix_gray = lept.pixConvertTo8(pix)
        check(pix_gray ~= nil, "pixConvertTo8()  NULL")
    end
    table.insert(temp_pixs, pix_gray)

    if options.save_intermediate_images then
        save_pix(pix_gray, "to_gray.bmp")
    end
    return pix_gray
end

local function filter_scale_up(pix, options, temp_pixs)
    if not options.scale_up or options.scale_factor == 1 then return pix end

    local depth = lept.pixGetDepth(pix)
    check(depth == 8 or depth == 1,
          "Scale filter (pixScale)  8  1- ,  " .. depth .. ". 'to_gray'    .")

    if options.debug then log("  ()  scale_factor: " .. options.scale_factor) end
    local pix_scaled = lept.pixScale(pix, options.scale_factor, options.scale_factor)

    check(pix_scaled ~= nil, "pixScale()  NULL.   : " .. depth)

    table.insert(temp_pixs, pix_scaled)
    if options.save_intermediate_images then
        save_pix(pix_scaled, "scale_up.bmp")
    end
    return pix_scaled
end

local function filter_equalize_hist(pix, options, temp_pixs)
    if not options.equalize_hist then return pix end
    if options.debug then log("  ") end
    local pix_eq = lept.pixEqualizeTRC(pix, 1.0, 10)
    check(pix_eq ~= nil, "pixEqualizeTRC()  NULL")

    if pix_eq ~= pix then
        table.insert(temp_pixs, pix_eq)
    end

    if options.save_intermediate_images then
        save_pix(pix_eq, "equalize_hist.bmp")
    end
    return pix_eq
end

local function filter_median(pix, options, temp_pixs)
    if not options.median then return pix end

    local depth = lept.pixGetDepth(pix)
    check(depth == 8 or depth == 1,
          "Median filter (pixRankFilter)  8  1- ,  " .. depth .. ". 'to_gray'    .")

    if options.debug then log("    : " .. options.median_size) end

    local size = options.median_size
    local pix_med = lept.pixRankFilter(pix, size, size, 0.5)

    if pix_med == nil then
        if options.debug then log("pixRankFilter  ,   ") end
        return pix
    end

    table.insert(temp_pixs, pix_med)

    if options.save_intermediate_images then
        save_pix(pix_med, "median.bmp")
    end
    return pix_med
end

local function filter_unsharp_mask(pix, options, temp_pixs)
    if not options.unsharp_mask then return pix end

    local depth = lept.pixGetDepth(pix)
    check(depth == 8, "Unsharp mask filter (pixUnsharpMaskingGray)  8- ,  " .. depth .. ". 'to_gray'    .")

    if options.debug then log("   (Unsharp Mask)  size=" .. options.unsharp_size .. ", fract=" .. options.unsharp_fract) end

    local pix_sharp = lept.pixUnsharpMaskingGray(pix, options.unsharp_size, ffi.cast("float", options.unsharp_fract))

    if pix_sharp == nil then
        if options.debug then log("pixUnsharpMaskingGray  ,   ") end
        return pix
    end

    table.insert(temp_pixs, pix_sharp)

    if options.save_intermediate_images then
        save_pix(pix_sharp, "unsharp_mask.bmp")
    end
    return pix_sharp
end

local function filter_invert(pix, options, temp_pixs)
    if not options.invert then return pix end
    if options.debug then log("  ") end

    local pix_inv = lept.pixInvert(nil, pix) -- nil  dest   Pix
    check(pix_inv ~= nil, "pixInvert()  NULL")

    table.insert(temp_pixs, pix_inv)

    if options.save_intermediate_images then
        save_pix(pix_inv, "invert.bmp")
    end
    return pix_inv
end

local function filter_binary(pix, options, temp_pixs)
    if not options.binary then return pix end
    if options.debug then log(" ") end

    local pix_bin
    local depth = lept.pixGetDepth(pix)
    check(depth == 8, "  8- ,  : " .. depth)

    --   Pix** (  )
    local ppixth = ffi.new("Pix*[1]", nil)
    local ppixd  = ffi.new("Pix*[1]", nil)

    if options.binary_method == "simple" then
        if options.debug then log("   (=" .. options.binary_threshold .. ")") end
        pix_bin = lept.pixThresholdToBinary(pix, options.binary_threshold)
        check(pix_bin ~= nil, "pixThresholdToBinary()  NULL")

    elseif options.binary_method == "otsu" then
        if options.debug then
            log("   (Otsu)  : tile=" .. options.otsu_tile_x .. "x" .. options.otsu_tile_y ..
                ", smooth=" .. options.otsu_smooth_x .. "x" .. options.otsu_smooth_y .. ", score=" .. options.otsu_score_fract)
        end

        pix_bin = lept.pixOtsuAdaptiveThreshold(pix,
            options.otsu_tile_x,
            options.otsu_tile_y,
            options.otsu_smooth_x,
            options.otsu_smooth_y,
            ffi.cast("float", options.otsu_score_fract),
            ppixth,
            ppixd)

        --    NULL,    
        if pix_bin == nil and ppixd[0] ~= nil then
            pix_bin = ppixd[0]
        end

        check(pix_bin ~= nil, "pixOtsuAdaptiveThreshold()  NULL.   (tile_x/y >= 16)   8- .")

    elseif options.binary_method == "sauvola" then
        if options.debug then
            log("   (Sauvola)  : whsize=" .. options.sauvola_whsize ..
                ", factor=" .. options.sauvola_factor .. ", nx=" .. options.sauvola_nx .. ", ny=" .. options.sauvola_ny)
        end

        pix_bin = lept.pixSauvolaBinarizeTiled(pix,
            options.sauvola_whsize,
            ffi.cast("float", options.sauvola_factor),
            options.sauvola_nx,
            options.sauvola_ny,
            ppixth,
            ppixd)

        --    NULL,    
        if pix_bin == nil and ppixd[0] ~= nil then
            pix_bin = ppixd[0]
        end

        check(pix_bin ~= nil, "pixSauvolaBinarizeTiled()  NULL.   (whsize >= 15)   8- .")

    else
        error(" binary_method: " .. options.binary_method)
    end

    --   Pix (   )
    if ppixth[0] ~= nil then lept.pixDestroy(ppixth) end
    -- ppixd[0]  pix_bin     cleanup  temp_pixs.
    if ppixd[0] ~= nil and ppixd[0] ~= pix_bin then lept.pixDestroy(ppixd) end

    table.insert(temp_pixs, pix_bin)
    if options.save_intermediate_images then
        save_pix(pix_bin, "binary.bmp")
    end
    return pix_bin
end

--   OCR
tess.ocr = function(image_address, width, height, length, data_path, language)
    local temp_pixs = {}
    local initial_pix
    if tess.options.debug then
        log(" OCR, image_address: " .. tostring(image_address) ..
            " width: " .. tostring(width) .. " height: " .. tostring(height) .. " length: " .. tostring(length))
    end
    data_path = data_path or tess.data_path
    language = language or tess.language

    check(image_address and type(image_address) == 'number', "  ")
    check(width > 0 and height > 0, "  : width=%d, height=%d", width, height)
    check(length > 0, "  : length=%d", length)

    for param, value in pairs({
        page_seg_mode = tess.options.page_seg_mode,
        ocr_engine_mode = tess.options.ocr_engine_mode,
        recognition_level = tess.options.recognition_level,
        preprocessing_mode = tess.options.preprocessing_mode,
        dotproduct = tess.options.dotproduct
    }) do
        if value then check_param(param, value) end
    end
    check_filter_params(tess.options)
    check_scale_factor(tess.options.scale_factor)
    check_font_id(tess.options.font_id)

    local final_width = width
    local final_height = height

    local success, result, api = pcall(function()
        if tess.options.debug then log("  ") end
        local image_data_ptr = ffi.cast('unsigned char*', image_address)
        check(image_data_ptr ~= nil, "     NULL")
        local max_buffer_size = height * length

        local width_step = length
        if tess.options.debug then log(" width_step: " .. width_step) end

        initial_pix = lept.pixCreate(width, height, 24)
        check(initial_pix ~= nil, "pixCreate()  NULL")
        local pix_data = lept.pixGetData(initial_pix)
        check(pix_data ~= nil, "pixGetData()  NULL")
        local wpl = lept.pixGetWpl(initial_pix)
        local aligned_bytes_per_line = wpl * 4
        if tess.options.debug then log("Wpl: " .. wpl .. ", aligned_bytes_per_line: " .. aligned_bytes_per_line) end
        check(aligned_bytes_per_line >= width_step, "    Pix  : wpl=" .. wpl .. ", width_step=" .. width_step)

        for y = 0, height - 1 do
            local src_offset = y * width_step
            local dst_offset = y * aligned_bytes_per_line
            if tess.options.debug and (y == 0 or y == math.floor(height / 2)) then
                log(string.format(" %d, src_offset: %d, dst_offset: %d", y, src_offset, dst_offset))
            end
            for x = 0, width - 1 do
                local src_idx = src_offset + x * 3
                local dst_idx = dst_offset + x * 3
                if src_idx + 2 < max_buffer_size then
                    pix_data[dst_idx] = image_data_ptr[src_idx + 2]   -- R
                    pix_data[dst_idx + 1] = image_data_ptr[src_idx + 1] -- G
                    pix_data[dst_idx + 2] = image_data_ptr[src_idx]     -- B
                    if tess.options.debug and y == 0 and x < 5 then
                        log(string.format("Pix  %d,0: R=%d, G=%d, B=%d", x, pix_data[dst_idx], pix_data[dst_idx + 1], pix_data[dst_idx + 2]))
                    end
                else
                    if tess.options.debug then log(string.format("   at y=%d, x=%d, src_idx=%d", y, x, src_idx)) end
                    pix_data[dst_idx] = 0
                    pix_data[dst_idx + 1] = 0
                    pix_data[dst_idx + 2] = 0
                end
            end
            local remainder = aligned_bytes_per_line - (width * 3)
            if remainder > 0 then
                for i = 0, remainder - 1 do
                    pix_data[dst_offset + width * 3 + i] = 0
                end
            end
        end

        if tess.options.debug then
            local depth = lept.pixGetDepth(initial_pix)
            log("Pix , width: " .. width .. ", height: " .. height .. ", depth: " .. depth)
        end

        if tess.options.save_intermediate_images then
            save_pix(initial_pix, "pre_filter_input.bmp")
        end

        local current_pix = initial_pix
        if tess.options.preprocessing_mode == "auto" then
            if tess.options.debug then log(" Tesseract   'auto'") end
            local results, api = run_tesseract(lept.pixGetData(current_pix), width, height, 3, width_step, data_path, language)
            return results, api
        end

        if tess.options.debug then log(" Tesseract   'manual'") end

        current_pix = filter_to_gray(current_pix, tess.options, temp_pixs)

        local filters = {
            scale_up = filter_scale_up,
            equalize_hist = filter_equalize_hist,
            median = filter_median,
            unsharp_mask = filter_unsharp_mask,
            invert = filter_invert,
            binary = filter_binary
        }

        local applied_filters = {}
        for _, filter_name in ipairs(tess.options.filter_order) do
            if not applied_filters[filter_name] and filters[filter_name] and tess.options[filter_name] then
                if tess.options.debug then
                    local depth_before = lept.pixGetDepth(current_pix)
                    log("  " .. filter_name .. ", : " .. depth_before)
                end
                if tess.options.debug then log(" : " .. filter_name) end
                current_pix = filters[filter_name](current_pix, tess.options, temp_pixs)
                applied_filters[filter_name] = true
                if current_pix == nil then
                    error(" " .. filter_name .. "  NULL")
                end
                if tess.options.debug then
                    local depth_after = lept.pixGetDepth(current_pix)
                    log("  " .. filter_name .. ", : " .. depth_after)
                end
            end
        end

        local depth = lept.pixGetDepth(current_pix)
        if tess.options.debug then log("  Pix: " .. depth) end

        local results, api = run_tesseract_pix(current_pix, data_path, language)

        if tess.options.return_coordinates and tess.options.scale_factor ~= 1 then
            local scale_factor = tess.options.scale_factor
            for i, result in ipairs(results) do
                result.left = math.floor(result.left / scale_factor)
                result.top = math.floor(result.top / scale_factor)
                result.right = math.floor(result.right / scale_factor)
                result.bottom = math.floor(result.bottom / scale_factor)
            end
        end

        return results, api
    end)

    cleanup(api, temp_pixs, initial_pix)

    if not success then
        if tess.options.debug then log("OCR   : " .. tostring(result)) end
        error(result)
    end

    if tess.options.debug then log("  OCR: " .. type(result)) end
    return result
end

return tess